import time

import pytest
from peewee import DoesNotExist

from median.base import BaseView, BaseViewException
from median.constant import EtatListe, PatientGlobal, TypeListe, TypeServiListe
# from median.views import Dispensation, Items, Dispensation_Suggestion
from median.models import Cip, Endpoint, Magasin, Poste, Product, Service, Ucd
from median.models import Stock, ListeModel, ListeItemModel, WorkItemModel, Seuil, Compteur
from median.views import Riedl, RiedlView
from datetime import datetime

# from datetime import datetime, date


@pytest.fixture()
def rectest():
    print("Running method level setUp")
    Product.delete().execute()
    WorkItemModel.delete().execute()
    ListeItemModel.delete().execute()
    ListeModel.delete().execute()
    rec_data = {}
    stk_data = []

    Compteur.get_or_create(cle="RIEDL_INVENTORY", defaults={
        "val": 1
    })

    pos99, _ = Poste.get_or_create(poste="RIEDL_99", defaults={
        "pc": "SERVEUR"
    })

    pos00, _ = Poste.get_or_create(poste="RIEDL_00", defaults={
        "pc": "SERVEUR"
    })

    whs99, _ = Magasin.get_or_create(mag="L99", defaults={
        "type_mag": pos99.poste,
        "eco_type": "L",
        "id_robot": "R01",
        "id_zone": "TMP",
    })

    whs00, _ = Magasin.get_or_create(mag="L00", defaults={
        "type_mag": pos00.poste,
        "eco_type": "L",
        "id_robot": "R00",
        "id_zone": "TMP",
    })

    enptlb99, _ = Endpoint.get_or_create(code="L99.LB01", defaults={
        "libelle": "Loading Bay 1 / Sortie urgence",
        "secteur": pos99.poste,
        "type_dest": "RIEDL",
    })

    enptex1_99, _ = Endpoint.get_or_create(code="L99.EX01", defaults={
        "libelle": "Exit 1 / Sortie 1",
        "secteur": pos99.poste,
        "type_dest": "RIEDL",
    })

    enptex2_99, _ = Endpoint.get_or_create(code="L99.EX02", defaults={
        "libelle": "Exit 2 / Sortie 2",
        "secteur": pos99.poste,
        "type_dest": "RIEDL",
    })

    ward_00, _ = Service.get_or_create(code="WARD00", defaults={
        "libelle": "Test Ward",
    })

    pr1 = Product()
    pr1.reference = "ECO6547"
    pr1.designation = "Eco Medoc"
    pr1.save()

    pr2 = Product()
    pr2.reference = "ECO9876"
    pr2.designation = "Eco Medoc 2"
    pr2.save()

    pr3 = Product()
    pr3.reference = "ECO8754"
    pr3.designation = "Eco Medoc 3"
    pr3.save()

    pr4 = Product()
    pr4.reference = "ECO6521"
    pr4.designation = "Eco Medoc 4"
    pr4.save()

    seuil1 = Seuil()
    seuil1.reference = pr1.reference
    seuil1.zone = pos00.poste
    seuil1.fraction = 100
    seuil1.save()

    ucd1 = Ucd()
    ucd1.ucd = "3400800000001"
    ucd1.reference = pr1.reference
    ucd1.save()

    cip1 = Cip()
    cip1.ucd = ucd1.ucd
    cip1.cip = "3400900000002"
    cip1.save()

    cip2 = Cip()
    cip2.ucd = ucd1.ucd
    cip2.cip = "3400900000003"
    cip2.save()

    # aurjourdhui = date.today()
    # expiration = datetime.date(2030, 10, 25)

    ul01 = ListeModel()
    ul01.liste = "%s-TEST-DONE" % time.strftime("%Y%m%d-%H%M%S")
    ul01.service = "DEENOVA"
    ul01.type_servi = TypeServiListe.RiedlBoite.value
    ul01.mode = TypeListe.Output.value
    ul01.etat = EtatListe.EnCours.value
    ul01.fusion = "TEST_PYTHON"
    ul01.zone_deb = "RIEDL"
    ul01.zone_fin = pos00.poste
    ul01.num_ipp = PatientGlobal.Ipp.value
    ul01.num_sej = PatientGlobal.Sejour.value
    ul01.save()

    it01 = ListeItemModel()
    it01.liste = ul01.liste
    it01.mode = ul01.mode
    it01.etat = EtatListe.Solde.value
    it01.reference = pr1.reference
    it01.item = "000001"
    it01.qte_dem = 100
    it01.qte_serv = 100
    it01.lot = "LOT000001"
    it01.info = "3400800000007"
    it01.ucd = "3400800000007"
    it01.cip = "3400900000009"
    it01.save()

    it02 = ListeItemModel()
    it02.liste = ul01.liste
    it02.mode = ul01.mode
    it02.etat = EtatListe.Solde.value
    it02.reference = pr2.reference
    it01.item = "000002"
    it02.qte_dem = 100
    it02.qte_serv = 110
    it02.lot = "LOT000002"
    it02.info = "3400800000014"
    it02.ucd = "3400800000014"
    it02.cip = "3400900000019"
    it02.save()

    it03 = ListeItemModel()
    it03.liste = ul01.liste
    it03.mode = ul01.mode
    it03.etat = EtatListe.Solde.value
    it03.reference = pr3.reference
    it03.item = "000003"
    it03.qte_dem = 120
    it03.qte_serv = 150
    it03.lot = "LOT000003"
    it03.info = "3400800000021"
    it03.ucd = "3400800000021"
    it03.cip = "3400900000029"
    it03.save()

    ul02 = ListeModel()
    ul02.liste = "%s-TEST-DRAFT" % time.strftime("%Y%m%d-%H%M%S")
    ul02.service = "DEENOVA"
    ul02.type_servi = TypeServiListe.RiedlBoite.value
    ul02.mode = TypeListe.Output.value
    ul02.etat = EtatListe.Vierge.value
    ul02.fusion = "TEST_PYTHON2"
    ul02.zone_deb = "RIEDL"
    ul02.zone_fin = pos00.poste
    ul02.num_ipp = PatientGlobal.Ipp.value
    ul02.num_sej = PatientGlobal.Sejour.value
    ul02.save()

    ul02it01 = ListeItemModel()
    ul02it01.liste = ul02.liste
    ul02it01.mode = ul02.mode
    ul02it01.etat = EtatListe.Vierge.value
    ul02it01.reference = pr1.reference
    ul02it01.item = "000001"
    ul02it01.qte_dem = 110
    ul02it01.qte_serv = 0
    ul02it01.lot = "LOT000001"
    ul02it01.info = "3400800000007"
    ul02it01.ucd = "3400800000007"
    ul02it01.cip = "3400900000009"
    ul02it01.save()

    for n in range(8):
        stk = Stock()
        stk.reference = pr1.reference
        stk.adresse = "L01"
        stk.magasin = whs00.mag
        stk.lot = "ABCDS"
        stk.quantite = 25
        stk.date_entree = datetime.now()
        stk.save()
        stk_data.append(stk)

    rec_data["riedl99"] = pos99
    rec_data["riedl00"] = pos00
    rec_data["whs99"] = whs99
    rec_data["whs00"] = whs00
    rec_data["enptlb99"] = enptlb99
    rec_data["enptex1_99"] = enptex1_99
    rec_data["enptex2_99"] = enptex2_99
    rec_data["ward_00"] = ward_00
    rec_data["pr1"] = pr1
    rec_data["pr2"] = pr2
    rec_data["pr3"] = pr3
    rec_data["pr4"] = pr4
    rec_data["ucd1"] = ucd1
    rec_data["cip1"] = cip1
    rec_data["ul01"] = ul01
    rec_data["ul02"] = ul02
    # rec_data['today'] = aurjourdhui
    # rec_data['expiry'] = expiration

    yield rec_data

    print("Running method level tearDown")
    for x in stk_data:
        x.delete_instance()
    ul02.delete_instance()
    ul01.delete_instance()
    cip1.delete_instance()
    cip2.delete_instance()
    ucd1.delete_instance()
    seuil1.delete_instance()
    pr4.delete_instance()
    pr3.delete_instance()
    pr2.delete_instance()
    pr1.delete_instance()
    ward_00.delete_instance()
    enptlb99.delete_instance()
    enptex1_99.delete_instance()
    enptex2_99.delete_instance()
    whs99.delete_instance()
    whs00.delete_instance()
    pos99.delete_instance()
    pos00.delete_instance()


class TestViewRiedl:
    def test_init_riedlview_without_poste(self, rectest):
        """
        Test d'initialisation de la classe
        """
        with pytest.raises(BaseViewException) as e:
            rdl = Riedl()
        assert "poste must be required" in str(e.value)

    def test_init_riedlview_unknown(self, rectest):
        """Instanciate Riedl with unknonw poste"""
        with pytest.raises(BaseViewException) as e:
            rdl = Riedl("UNKNOWN")
        assert "does not exists" in str(e.value)

    def test_init_riedlview(self, rectest):
        """Test instance with known Riedl"""
        rdl = Riedl(rectest["riedl99"].poste)

    def test_riedlview_warehouse_99(self, rectest):
        """Retrieve warehouse associat to a Riedl"""
        rdl = Riedl(rectest["riedl99"].poste)
        assert len(rdl.warehouses) == 1
        assert rdl.warehouses[0].mag == rectest["whs99"].mag

    def test_riedlview_warehouse_00(self, rectest):
        """Retrieve warehouse associat to a Riedl"""
        rdl = Riedl(rectest["riedl00"].poste)
        assert len(rdl.warehouses) == 1
        assert rdl.warehouses[0].mag == rectest["whs00"].mag

    def test_riedlview_endpoints(self, rectest):
        """Retrieve warehouse associat to a Riedl"""
        rdl = Riedl(rectest["riedl99"].poste)
        assert len(rdl.endpoints) == 3

    def test_riedlview_add_endpoint(self, rectest):
        """Add new endpoint"""
        rdl = Riedl(rectest["riedl00"].poste)
        assert rdl.poste.poste == rectest["riedl00"].poste
        assert len(rdl.warehouses) == 1
        assert rdl.warehouses[0].mag == rectest["whs00"].mag
        epc = rdl.add_endpoint("L00.LB01", "Loading bay 1", 1, 1)
        assert len(rdl.endpoints) == 1
        epc1 = rdl.add_endpoint("L00.EX01", "Exit 1", 1, 1)
        assert len(rdl.endpoints) == 2
        epc2 = rdl.add_endpoint("L00.EX02", "Exit 2", 1, 1)
        assert len(rdl.endpoints) == 3
        epc1.delete_instance()
        epc2.delete_instance()
        epc.delete_instance()

    def test_compute_checksum(self, rectest):
        rdl = Riedl(rectest["riedl00"].poste)
        chksum = rdl._compute_checksum("2330010000003")
        assert chksum == "3"

    def test_compute_checksum(self, rectest):
        rdl = Riedl(rectest["riedl00"].poste)
        chksum = rdl._compute_checksum("233001000000")
        assert chksum == "3"

    def test_riedlview_create_unknown_ward_return(self, rectest):
        rdl = Riedl(rectest["riedl00"].poste)
        with pytest.raises(BaseViewException) as e:
            rdl.create_ward_return(
                "UNKNOWN", rectest["pr1"].reference, 1, "LOTtest", ""
            )
        assert "does not exists" in str(e.value)

    def test_riedlview_unknown_drugs_return(self, rectest):
        rdl = Riedl(rectest["riedl00"].poste)
        with pytest.raises(BaseViewException) as e:
            rdl.create_ward_return(rectest["ward_00"].code, "UNKNOWN", 1, "LOTtest", "")
        assert "does not exists" in str(e.value)

    def test_riedlview_quantity_zero__return(self, rectest):
        rdl = Riedl(rectest["riedl00"].poste)
        with pytest.raises(BaseViewException) as e:
            rdl.create_ward_return(
                rectest["ward_00"].code, rectest["pr1"].reference, 0, "LOTtest", ""
            )
        assert "Quantity must be greather than 0" in str(e.value)

    def test_riedlview_create_ward_return(self, rectest):
        rdl = RiedlView(rectest["riedl00"].poste)
        lstId = rdl.create_ward_return(
            rectest["ward_00"].code,
            rectest["pr1"].reference,
            1,
            "LOTtest",
            None,
            rectest["ucd1"].ucd,
        )
        assert lstId.zone_fin == rectest["riedl00"].poste
        assert lstId.nb_item == 1
        assert lstId.service == rectest["ward_00"].code
        assert lstId.liste.startswith(rectest["whs00"].mag)
        assert lstId.liste.startswith("%s-%s" % (rectest["whs00"].mag, rectest["ward_00"].code))

        itm1 = ListeItemModel.get(liste=lstId.liste, mode=lstId.mode, item='000001')
        assert itm1.reference == rectest["pr1"].reference
        assert itm1.qte_dem == 1
        assert itm1.lot == "LOTTEST"
        assert itm1.contenant.startswith('233001')
        assert itm1.info.startswith('34008')
        assert len(itm1.info) == 13

        lstId2 = rdl.create_ward_return(
            rectest["ward_00"].code,
            rectest["pr1"].reference,
            1,
            "LOTtest",
            None,
            rectest["ucd1"].ucd,
        )
        assert lstId2.liste == lstId.liste
        assert lstId2.zone_fin == rectest["riedl00"].poste

        assert lstId2.nb_item == 2

        lstId.delete_instance()

    def test_riedlview_create_taking_mode_list_without_model(self, rectest):
        """
        Create a taking mode without
        """
        rdl = Riedl(rectest["riedl00"].poste)
        with pytest.raises(BaseViewException) as e:
            rdl.create_taking_mode()
        assert "Id of the model is required" in str(e.value)

    def test_riedlview_create_taking_mode_list_with_unknown_model(self, rectest):
        """
        Create a taking mode with unknown model
        """
        rdl = Riedl(rectest["riedl00"].poste)
        with pytest.raises(BaseViewException) as e:
            rdl.create_taking_mode(1)
        assert "List not found:" in str(e.value)

    # /!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\
    # /!\ THIS TEST IS DISABLE, BECAUSE LIST CAN BE DRAFT, IN PROGRESS, DONE  /!\
    # /!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\
    #
    # def test_riedlview_create_taking_mode_list_undone_model(self, rectest):
    #     """
    #     Create a taking mode with undone model
    #     """
    #     rdl = Riedl(rectest["riedl00"].poste)
    #     with pytest.raises(BaseViewException) as e:
    #         rdl.create_taking_mode(rectest["ul01"].pk)
    #     assert "List must be done:" in str(e.value)

    def test_riedlview_create_taking_mode_list_model_valid(self, rectest):
        """
        Create a taking mode with undone model
        """
        rdl = Riedl(rectest["riedl00"].poste)

        rectest["ul01"].etat = EtatListe.Solde.value
        rectest["ul01"].save()

        tk1 = rdl.create_taking_mode(rectest["ul01"].pk)

        assert tk1.pk > 0
        assert tk1.mode == TypeListe.Input.value
        assert tk1.etat == EtatListe.Vierge.value
        assert tk1.liste.endswith("-P")
        assert tk1.fusion == "PRELEVEMENT"
        assert tk1.service == rectest["ul01"].service
        assert tk1.nb_item == 2

        tk1.delete_instance()

    def test_riedlview_create_taking_mode_list_check_lines(self, rectest):
        """
        Create a taking mode and check lines created
        """
        rdl = Riedl(rectest["riedl00"].poste)

        rectest["ul01"].etat = EtatListe.Solde.value
        rectest["ul01"].save()

        tk1 = rdl.create_taking_mode(rectest["ul01"].pk)

        assert tk1.pk > 0
        assert tk1.nb_item == 2

        itm1 = ListeItemModel.get(liste=tk1.liste, mode=tk1.mode, item='000001')
        assert itm1.reference == rectest["pr2"].reference
        assert itm1.qte_dem == 10
        assert itm1.lot == "LOT000002"
        assert itm1.contenant.startswith('255001')
        assert itm1.info.startswith('34008')
        assert itm1.ucd.startswith('34008')
        assert itm1.cip.startswith('34009')
        assert len(itm1.info) == 13

        itm2 = ListeItemModel.get(liste=tk1.liste, mode=tk1.mode, item='000002')
        assert itm2.reference == rectest["pr3"].reference
        assert itm2.qte_dem == 30
        assert itm2.lot == "LOT000003"
        assert itm2.contenant.startswith('255001')
        assert itm2.info.startswith('34008')
        assert itm2.ucd.startswith('34008')
        assert itm2.cip.startswith('34009')
        assert len(itm2.info) == 13

        # GTIN must be unique
        assert itm1.contenant != itm2.contenant

        tk1.delete_instance()

    def test_riedlview_generate_inventory_l00(self, rectest):
        rdl = RiedlView(rectest["riedl00"].poste)
        list_id, nb_ref = rdl.generate_inventory()

        assert list_id.pk > 0
        assert nb_ref == 2

        list_id.delete_instance()

    def test_riedlview_generate_inventory_l99(self, rectest):
        rdl = RiedlView(rectest["riedl99"].poste)
        list_id, nb_ref = rdl.generate_inventory()

        assert list_id.pk > 0
        assert nb_ref == 0

        list_id.delete_instance()
