import time

import pytest
from peewee import DoesNotExist

from median.base import BaseView, BaseViewException
from median.constant import EtatListe, PatientGlobal, TypeListe, TypeServiListe, ExternalMachineType, PickingMachineType, CuttingMachineType
from median.models import Cip, Endpoint, Magasin, Poste, Product, Service, Ucd, Cip, Adresse, Gtin
from median.models import Stock, ListeModel, ListeItemModel, Gpao, Historique, Espace
from median.views import ReplenishmentView
from median.utils import padding_address, date_add_month
from datetime import datetime


class TestViewReplenishmentStandalone:

    def setup_method(self, method):
        ListeItemModel.delete().execute()
        ListeModel.delete().execute()
        Gpao.delete().execute()
        Historique.delete().execute()
        self.mag_1, _ = Magasin.get_or_create(mag="ST1", type_mag="ACCED_ST1", defaults={
            "eco_type": "C",
            "type_machine": "ACCED_V2",
            "id_zone": 1000,
            "id_robot": 2000
        })
        self.pro_dol, _ = Product.get_or_create(reference="DOLIPRANE", defaults={
            "designation": "Doliprane",
            "com_med": "Test reappro",
            "reap_mode": "T"
        })
        self.pro_ser10, _ = Product.get_or_create(reference="SERESTA10", defaults={
            "designation": "Doliprane",
            "com_med": "Test reappro",
            "reap_mode": "E"
        })

    def teardown_method(self, method):
        Magasin.delete().execute()
        Product.delete().where(Product.reference==self.pro_dol.reference).execute()
        Product.delete().where(Product.reference==self.pro_ser10.reference).execute()

    def test_without_warehouse(self):
        with pytest.raises(BaseViewException) as e:
            ReplenishmentView()
        assert "please provide warehouse code" in str(e)

    def test_unknown_warehouse(self):
        with pytest.raises(BaseViewException) as e:
            ReplenishmentView("ABC")
        assert "Warehouse code" in str(e)

    def test_init_replenishmentview(self):
        reappro = ReplenishmentView("ST1")

    def test_replenishment_unknown_product(self):
        with pytest.raises(BaseViewException) as e:
            reappro = ReplenishmentView("ST1")
            reappro.add_reference("UNKNOWN", 100)
        assert "Product UNKNOWN" in str(e)

    def test__replenishmentview_external_relate(self):
        reappro = ReplenishmentView("ST1")
        assert not reappro._is_relate_to_external(),'ST1 is not relate to external'

    def test_replenishment_add_product(self):
        reappro = ReplenishmentView("ST1", "Pytest")
        reappro.add_reference(self.pro_dol.reference, 100)

        head = ListeModel.select(ListeModel).where(
            ListeModel.mode==TypeListe.Input.value,
            ListeModel.zone_fin==self.mag_1.type_mag
        )
        assert len(head) == 1

        item = ListeItemModel.select(ListeItemModel).where(
            ListeItemModel.mode==TypeListe.Input.value,
            ListeItemModel.reference==self.pro_dol.reference
        )
        assert len(item) == 1
        assert item[0].qte_dem == 100

        gp = Gpao.select(Gpao).where(
            Gpao.ref==self.pro_dol.reference, Gpao.type_mvt=="R"
        )
        assert len(gp) == 1, "Must have one gpao line"

        his = Historique.select(Historique).where(
            Historique.type_mouvement=="REF",
            Historique.reference==self.pro_dol.reference
        )
        assert len(his) == 1, "Must have one history line"

    def test_replenishment_add_two_product(self):
        reappro = ReplenishmentView("ST1", "Pytest")
        reappro.add_reference(self.pro_dol.reference, 100)
        reappro.add_reference(self.pro_ser10.reference, 45, 50)

        head = ListeModel.select(ListeModel).where(
            ListeModel.mode==TypeListe.Input.value,
            ListeModel.zone_fin==self.mag_1.type_mag
        )
        assert len(head) == 1
        assert head[0].nb_item == 2

        items = ListeItemModel.select(ListeItemModel).where(
            ListeItemModel.mode==TypeListe.Input.value,
            ListeItemModel.liste==head[0].liste
        )

        assert len(items) == 2, "We need to have 2 items"

        item = ListeItemModel.select(ListeItemModel).where(
            ListeItemModel.mode==TypeListe.Input.value,
            ListeItemModel.reference==self.pro_dol.reference
        )
        assert len(item) == 1
        assert item[0].qte_dem == 100
        assert item[0].item == "00001"
        assert item[0].fraction == 100

        item = ListeItemModel.select(ListeItemModel).where(
            ListeItemModel.mode==TypeListe.Input.value,
            ListeItemModel.reference==self.pro_ser10.reference
        )
        assert len(item) == 1
        assert item[0].qte_dem == 45
        assert item[0].item == "00002"
        assert item[0].fraction == 50

        gp = Gpao.select(Gpao).where(
            Gpao.type_mvt=="R"
        )
        assert len(gp) == 2, "Must have two gpao line"

        his_lines = Historique.select(Historique).where(
            Historique.type_mouvement=="REF",
        )
        assert len(his_lines) == 2, "Must have two history line"

        his = Historique.select(Historique).where(
            Historique.type_mouvement=="REF",
            Historique.reference==self.pro_dol.reference
        )
        assert len(his) == 1, "Must have one history line"
        assert his[0].quantite_mouvement == 100, 'Quantity must be 100'


class TestViewReplenishmentWithExterne:

    def setup_method(self, method):
        ListeItemModel.delete().execute()
        ListeModel.delete().execute()
        Gpao.delete().execute()
        Historique.delete().execute()
        self.mag_1, _ = Magasin.get_or_create(mag="ST1", type_mag="ACCED_ST1", defaults={
            "eco_type": "C",
            "type_machine": PickingMachineType.Acced_Boite_Pass.value,
            "id_zone": 1000,
            "id_robot": 2000
        })
        self.mag_ext1, _ = Magasin.get_or_create(mag="EX1", type_mag="EXTERNE_1", defaults={
            "eco_type": "E",
            "type_machine": ExternalMachineType.Etagere.value,
            "id_zone": 1000,
            "id_robot": 2000
        })
        self.mag_ext2, _ = Magasin.get_or_create(mag="EX2", type_mag="EXTERNE_2", defaults={
            "eco_type": "E",
            "type_machine": ExternalMachineType.Etagere.value,
            "id_zone": 1080,
            "id_robot": 2080
        })
        self.pro_dol, _ = Product.get_or_create(reference="DOLIPRANE", defaults={
            "designation": "Doliprane",
            "com_med": "Test reappro",
            "reap_mode": "T"
        })
        self.pro_ser10, _ = Product.get_or_create(reference="SERESTA10", defaults={
            "designation": "Seresta 10mg",
            "com_med": "Test reappro",
            "reap_mode": "T"
        })
        self.pro_kar75, _ = Product.get_or_create(reference="KARDEGIC75", defaults={
            "designation": "Kardegic 75 mg",
            "com_med": "Test reappro",
            "reap_mode": "E"
        })
        Espace.get_or_create(poste=self.mag_1.type_mag, mag=self.mag_1.mag, defaults={
            "alloc": 1,
            "service": 1,
            "gest": "gest_trad",
            "sub_gest": "a"
        })
        Espace.get_or_create(poste=self.mag_ext1.type_mag, mag=self.mag_1.mag, defaults={
            "alloc": 0,
            "service": 1,
            "gest": "gest_trad",
            "sub_gest": "a"
        })
        Espace.get_or_create(poste=self.mag_ext2.type_mag, mag=self.mag_1.mag, defaults={
            "alloc": 0,
            "service": 1,
            "gest": "gest_trad",
            "sub_gest": "a"
        })
        for eq in [self.mag_ext1.mag, self.mag_ext2.mag]:
            for adr in range(9):
                front = padding_address([eq, '1', '1', adr + 1, '1'])
                Adresse.get_or_create(adresse=front, defaults={
                    "etat": "L",
                    "format": "BOITE PASS",
                    "magasin": eq,
                    "contenant": "%07d" % (2345 * adr),
                })
        stk_contenant = "12345678"
        stk_adr = padding_address([self.mag_ext1.mag, '1', '1', 2, '1'])
        Stock.get_or_create(
            adresse=stk_adr, reference=self.pro_dol.reference, defaults={
                "quantite": 500,
                "lot": "ABCDEF1234",
                "date_peremption": date_add_month(datetime.now(), 10),
                "contenant": stk_contenant,
                "magasin": self.mag_ext1.mag,
            }
        )
        Adresse.update({"etat": "O", "contenant": stk_contenant}).where(Adresse.adresse==stk_adr).execute()
        stk_contenant = "23456789"
        stk_adr = padding_address([self.mag_ext1.mag, '1', '1', 3, '1'])
        Stock.get_or_create(
            adresse=stk_adr, reference=self.pro_kar75.reference, defaults={
                "quantite": 300,
                "lot": "ABCDEF1234",
                "date_peremption": date_add_month(datetime.now(), 11),
                "contenant": stk_contenant,
                "magasin": self.mag_ext1.mag,
            }
        )
        Adresse.update({"etat": "O", "contenant": stk_contenant}).where(Adresse.adresse==stk_adr).execute()

    def teardown_method(self, method):
        Espace.delete().execute()
        Magasin.delete().execute()
        Product.delete().where(Product.reference==self.pro_dol.reference).execute()
        Product.delete().where(Product.reference==self.pro_ser10.reference).execute()
        Adresse.delete().execute()
        Stock.delete().execute()

    def test_init_replenishmentview_mag_one(self):
        reappro = ReplenishmentView(self.mag_ext1.mag)

    def test_init_replenishmentview_mag_two(self):
        reappro = ReplenishmentView(self.mag_ext2.mag)

    def test__replenishmentview_external_relate(self):
        reappro = ReplenishmentView(self.mag_1.mag)
        assert reappro._is_relate_to_external(),'ST1 is relate to EX1 and EX2'

    def test_replenishment_relations(self):
        reappro = ReplenishmentView(self.mag_1.mag)
        relations = reappro.relate_espace()
        assert len(relations) == 3, 'Must be relate to EX1, EX2 and ST1'
        assert self.mag_1.type_mag in [x.poste for x in relations]
        assert self.mag_ext1.type_mag in [x.poste for x in relations]
        assert self.mag_ext2.type_mag in [x.poste for x in relations]

    def test_replenishment_with_external_doliprane(self):
        reappro = ReplenishmentView("ST1", "Pytest")
        reappro.add_reference(self.pro_dol.reference, 100)

        head_in = ListeModel.select(ListeModel).where(
            ListeModel.mode==TypeListe.Input.value,
            ListeModel.zone_fin==self.mag_1.type_mag
        )
        assert len(head_in) == 0, 'No cutting list must be available'

        head_out = ListeModel.select(ListeModel).where(
            ListeModel.mode==TypeListe.Output.value,
            ListeModel.zone_fin==self.mag_1.type_mag
        )
        assert len(head_out) == 1, 'Output list must be available'

        item = ListeItemModel.select(ListeItemModel).where(
            ListeItemModel.mode==TypeListe.Output.value,
            ListeItemModel.reference==self.pro_dol.reference
        )
        assert len(item) == 1
        assert item[0].type_servi == TypeServiListe.ExterneBoite.value
        assert item[0].qte_dem == 100
        assert item[0].fraction == 100

    def test_replenishment_with_external_doliprane_kardegic(self):
        reappro = ReplenishmentView("ST1", "Pytest")
        reappro.add_reference(self.pro_dol.reference, 100)

        head_in = ListeModel.select(ListeModel).where(
            ListeModel.mode==TypeListe.Input.value,
            ListeModel.zone_fin==self.mag_1.type_mag
        )
        assert len(head_in) == 0, 'No cutting list must be available'

        head_out = ListeModel.select(ListeModel).where(
            ListeModel.mode==TypeListe.Output.value,
            ListeModel.zone_fin==self.mag_1.type_mag
        )
        assert len(head_out) == 1, 'Output list must be available'

        reappro.add_reference(self.pro_kar75.reference, 99)
        item = ListeItemModel.select(ListeItemModel).where(
            ListeItemModel.mode==TypeListe.Output.value,
            ListeItemModel.reference==self.pro_kar75.reference
        )
        assert len(item) == 1
        assert item[0].type_servi == TypeServiListe.ExterneBoite.value
        assert item[0].qte_dem == 99
        assert item[0].fraction == 100

    def test_replenishment_with_external_internal(self):
        reappro = ReplenishmentView("ST1", "Pytest")
        reappro.add_reference(self.pro_dol.reference, 100)

        head_in = ListeModel.select(ListeModel).where(
            ListeModel.mode==TypeListe.Input.value,
            ListeModel.zone_fin==self.mag_1.type_mag
        )
        assert len(head_in) == 0, 'No cutting list must be available'

        head_out = ListeModel.select(ListeModel).where(
            ListeModel.mode==TypeListe.Output.value,
            ListeModel.zone_fin==self.mag_1.type_mag
        )
        assert len(head_out) == 1, 'Output list must be available'

        # Now we add a product not available on external
        reappro.add_reference(self.pro_ser10.reference, 35)
        head_in = ListeModel.select(ListeModel).where(
            ListeModel.mode==TypeListe.Input.value,
            ListeModel.zone_fin==self.mag_1.type_mag
        )
        assert len(head_in) == 1, 'A cutting list must be available'

        item = ListeItemModel.select(ListeItemModel).where(
            ListeItemModel.mode==TypeListe.Input.value,
            ListeItemModel.reference==self.pro_ser10.reference
        )
        assert len(item) == 1
        assert item[0].type_servi == TypeServiListe.GlobaleBoite.value
        assert item[0].qte_dem == 35
        assert item[0].fraction == 100

class TestViewReplenishmentExterne:

    def setup_method(self, method):
        ListeItemModel.delete().execute()
        ListeModel.delete().execute()
        Gpao.delete().execute()
        Historique.delete().execute()
        self.mag_cut1, _ = Magasin.get_or_create(mag="DC1", type_mag="COUPE_1", defaults={
            "eco_type": "T",
            "type_machine": CuttingMachineType.Aide_Cut.value,
            "id_zone": 3000,
            "id_robot": 4000
        })
        self.mag_ext1, _ = Magasin.get_or_create(mag="EX1", type_mag="EXTERNE_1", defaults={
            "eco_type": "E",
            "type_machine": ExternalMachineType.Etagere.value,
            "id_zone": 1000,
            "id_robot": 2000
        })

        self.pro_dol, _ = Product.get_or_create(reference="DOLIPRANE", defaults={
            "designation": "Doliprane",
            "com_med": "Test reappro",
            "reap_mode": "T"
        })
        self.pro_ser10, _ = Product.get_or_create(reference="SERESTA10", defaults={
            "designation": "Seresta 10mg",
            "com_med": "Test reappro",
            "reap_mode": "E"
        })
        self.pro_kar75, _ = Product.get_or_create(reference="KARDEGIC75", defaults={
            "designation": "Kardegic 75 mg",
            "com_med": "Test reappro",
            "reap_mode": "E"
        })
        self.ucd_dol, _ = Ucd.get_or_create(reference=self.pro_dol.reference, defaults={
            "ucd": "3400892390918"
        })
        self.cip_dol, _ = Gtin.get_or_create(ucd=self.ucd_dol.ucd, defaults={
            "cip": "3400956369553",
            "dossier": 1,
            "qt_boite": 100,
            "qt_blister": 10,
            "qt_pass": 30
        })
        Espace.get_or_create(poste=self.mag_cut1.type_mag, mag=self.mag_cut1.mag, defaults={
            "alloc": 1,
            "service": 1,
            "gest": "gest_trad",
            "sub_gest": "a"
        })
        Espace.get_or_create(poste=self.mag_cut1.type_mag, mag=self.mag_ext1.mag, defaults={
            "alloc": 0,
            "service": 1,
            "gest": "gest_trad",
            "sub_gest": "a"
        })

    def teardown_method(self, method):
        Espace.delete().execute()
        Magasin.delete().execute()
        Product.delete().where(Product.reference==self.pro_dol.reference).execute()
        Product.delete().where(Product.reference==self.pro_ser10.reference).execute()
        Adresse.delete().execute()
        Stock.delete().execute()

    def test_init_replenishmentview_mag_external(self):
        reappro = ReplenishmentView(self.mag_ext1.mag)

    def test_init_replenishmentview_cutting_drugs(self):
        reappro = ReplenishmentView(self.mag_ext1.mag, "Pytest Ext")
        # We add drugs produce by AIDE CUT
        reappro.add_reference(self.pro_dol.reference, 90)

        head = ListeModel.select(ListeModel).where(
            ListeModel.mode==TypeListe.Input.value,
            ListeModel.zone_fin==self.mag_ext1.type_mag,
            ListeModel.type_servi==TypeServiListe.GlobaleBoite.value
        )
        assert len(head) == 1
        assert head[0].fusion == "REASSORT"

        item = ListeItemModel.select(ListeItemModel).where(
            ListeItemModel.mode==TypeListe.Input.value,
            ListeItemModel.reference==self.pro_dol.reference,
            ListeItemModel.type_servi==TypeServiListe.GlobaleBoite.value
        )
        assert len(item) == 1
        assert item[0].qte_dem == 90

        # We add drugs not produce by AIDE CUT (check cutting file)
        reappro.add_reference(self.pro_ser10.reference, 30)
        head = ListeModel.select(ListeModel).where(
            ListeModel.mode==TypeListe.Input.value,
            ListeModel.zone_fin==self.mag_ext1.type_mag,
            ListeModel.type_servi==TypeServiListe.ExterneBoite.value
        )
        assert len(head) == 1

        item = ListeItemModel.select(ListeItemModel).where(
            ListeItemModel.mode==TypeListe.Input.value,
            ListeItemModel.reference==self.pro_ser10.reference,
            ListeItemModel.type_servi==TypeServiListe.ExterneBoite.value
        )
        assert len(item) == 1
        assert item[0].qte_dem == 30

        gp = Gpao.select(Gpao).where(
            Gpao.ref==self.pro_ser10.reference, Gpao.type_mvt=="R"
        )
        assert len(gp) == 1, "Must have one gpao line"
        assert gp[0].qte == 30
        assert gp[0].fraction == 100

        his = Historique.select(Historique).where(
            Historique.type_mouvement=="REF",
            Historique.reference==self.pro_ser10.reference
        )
        assert len(his) == 1, "Must have one history line"
        assert his[0].quantite_mouvement == 30
        assert his[0].fraction == 100
