# flake8: noqa
import pytest
from peewee import DoesNotExist
from median.models import Patient, Prescription, Service, Product
from median.models import WorkItemModel, ListeErrorModel, ListeItemModel, ListeModel
from median.constant import PatientGlobal, TypeServiListe, TypeListe, EtatListe
from median.base import MedianLibException
import time
from datetime import datetime


@pytest.fixture()
def rectest():
    print("Running method level setUp")
    rec_data = {}

    pat_test = Patient(nom="Test", prenom="Patrick")
    pat_test.ipp = "ABCD1234"
    pat_test.sexe = "M"
    pat_test.date_maj = datetime.now()
    pat_test.save()

    presc = Prescription()
    presc.ordre = "Prescription 1"
    presc.ipp = "ECO1234"
    presc.sejour = "SEJ1234"
    presc.nom = "Test"
    presc.prenom = "Utilsiateur"
    presc.save()

    serv = Service()
    serv.code = "ECOTEST"
    serv.libelle = "Service de test"
    serv.ordre = 0
    serv.type_dest = "SERVICE"
    serv.moment_deb = "matin"
    serv.deb_matin = 7
    serv.deb_midi = 12
    serv.deb_soir = 18
    serv.deb_coucher = 21
    serv.nb_jour = 7
    serv.save()

    p1 = Product(reference="WKIP0001", designation="Produit Test 1")
    p1.save()
    rec_data["p1"] = p1

    p2 = Product(reference="WKIP0002", designation="Produit Test 2")
    p2.save()
    rec_data["p2"] = p2

    rec_data["patient"] = pat_test
    rec_data["prescription"] = presc
    rec_data["service"] = serv
    yield rec_data

    print("Running method level tearDown")
    p1.delete_instance()
    p2.delete_instance()
    pat_test.delete_instance()
    presc.delete_instance()
    serv.delete_instance()


class TestModelDispensation:
    def setup_method(self, method):
        ListeModel.delete().execute()
        ListeItemModel.delete().execute()
        WorkItemModel.delete().execute()
        ListeErrorModel.delete().execute()

    def test_unknown_dispensation(self):
        """Test access unknown dispensation"""
        with pytest.raises(DoesNotExist):
            ListeModel.get(liste="INCONNU")

    def test_dispensation_out_wo_prescription(self, rectest):
        """Test create a dispensation without precription"""
        with pytest.raises(ValueError):
            dis = ListeModel()
            dis.liste = "TEST dispensation"
            dis.mode = TypeListe.Output.value
            dis.etat = EtatListe.Vierge.value
            dis.type_servi = TypeServiListe.Nominatif.value
            dis.save()

    def test_dispensation_out_wo_prescription_exotique(self, rectest):
        """Test create a dispensation without precription, exoticque mode"""
        with pytest.raises(ValueError):
            dis = ListeModel()
            dis.liste = "TEST dispensation"
            dis.mode = TypeListe.Output.value
            dis.etat = EtatListe.Vierge.value
            dis.type_servi = TypeServiListe.Exotique.value
            dis.save()

    def test_dispensation_out_wo_patient(self, rectest):
        """Test create a dispensation without patient"""
        with pytest.raises(ValueError):
            dis = ListeModel()
            dis.liste = "TEST dispensation"
            dis.mode = TypeListe.Output.value
            dis.etat = EtatListe.Vierge.value
            dis.type_servi = TypeServiListe.Nominatif.value
            dis.id_prescription = rectest["prescription"]
            dis.save()

    def test_create_dispensation_out(self, rectest):
        """Test create a dispensation"""
        dis = ListeModel()
        dis.liste = "TEST dispensation"
        dis.mode = TypeListe.Output.value
        dis.etat = EtatListe.Vierge.value
        dis.type_servi = TypeServiListe.Nominatif.value
        dis.id_prescription = rectest["prescription"].pk
        dis.num_ipp = rectest["patient"].ipp
        dis.num_sej = 'SEJ1234'
        dis.save()
        assert dis.pk > 0
        dis.delete_instance()

    def test_create_bad_output_lot(self, rectest):
        """Test create an output lot list"""
        with pytest.raises(ValueError):
            dis = ListeModel()
            dis.liste = "LOT A SORTIR xxx"
            dis.fusion = "SORTIE DE LOTS"
            dis.service = rectest["service"].code
            dis.mode = TypeListe.Output.value
            dis.etat = EtatListe.Vierge.value
            dis.save()

    def test_create_output_lot(self, rectest):
        """Test create an output lot list"""
        dis = ListeModel()
        dis.liste = "LOT A SORTIR xxx"
        dis.fusion = "SORTIE DE LOTS"
        dis.service = rectest["service"].code
        dis.type_servi = TypeServiListe.GlobaleBoite.value
        dis.num_ipp = PatientGlobal.Ipp.value
        dis.num_sej = PatientGlobal.Sejour.value
        dis.id_servi = 2
        dis.mode = TypeListe.Output.value
        dis.etat = EtatListe.Vierge.value
        dis.save()
        assert dis.pk > 0
        dis.delete_instance()

    def test_riedl_create_output_lot(self, rectest):
        """Test create an output lot list"""
        dis = ListeModel()
        dis.liste = "RIEDL LOT OUTPUT"
        dis.fusion = "SORTIE DE LOTS"
        dis.service = rectest["service"].code
        dis.type_servi = TypeServiListe.RiedlBoite.value
        dis.num_ipp = PatientGlobal.Ipp.value
        dis.num_sej = PatientGlobal.Sejour.value
        dis.id_servi = 2
        dis.mode = TypeListe.Output.value
        dis.etat = EtatListe.Vierge.value
        dis.save()
        assert dis.pk > 0
        assert dis.zone_deb == "RIEDL"
        dis.delete_instance()

    def test_create_output_lot_wo_state(self, rectest):
        """Test create an output lot list without state"""
        dis = ListeModel()
        dis.liste = "LOT A SORTIR xxx"
        dis.fusion = "SORTIE DE LOTS"
        dis.service = rectest["service"].code
        dis.type_servi = TypeServiListe.GlobaleBoite.value
        dis.num_ipp = PatientGlobal.Ipp.value
        dis.num_sej = PatientGlobal.Sejour.value
        dis.id_servi = 2
        dis.mode = TypeListe.Output.value
        dis.save()
        assert dis.pk > 0
        assert dis.etat == EtatListe.Vierge.value
        dis.delete_instance()

    def test_output_expiry_wo_type_servi_erreur(self, rectest):
        """Test bad output of product expiry"""
        with pytest.raises(ValueError):
            dis = ListeModel()
            dis.liste = "PERIME A SORTIR xxx"
            dis.fusion = "SORTIE DE PERIMES"
            dis.service = rectest["service"].code
            dis.mode = TypeListe.Output.value
            dis.etat = "V"
            dis.save()

    def test_global_output_wo_ipp(self, rectest):
        """Test bad output of product expiry"""
        with pytest.raises(ValueError):
            dis = ListeModel()
            dis.liste = "PERIME A SORTIR xxx"
            dis.fusion = "SORTIE DE PERIMES"
            dis.service = rectest["service"].code
            dis.type_servi = TypeServiListe.GlobaleBoite.value
            dis.id_servi = 2
            dis.mode = TypeListe.Output.value
            dis.save()
            assert dis.pk > 0
            dis.delete_instance()

    def test_global_output_wo_episode(self, rectest):
        """Test bad output of product expiry"""
        with pytest.raises(ValueError):
            dis = ListeModel()
            dis.liste = "PERIME A SORTIR xxx"
            dis.fusion = "SORTIE DE PERIMES"
            dis.service = rectest["service"].code
            dis.type_servi = TypeServiListe.GlobaleBoite.value
            dis.num_ipp = PatientGlobal.Ipp.value
            dis.id_servi = 2
            dis.mode = TypeListe.Output.value
            dis.save()
            assert dis.pk > 0
            dis.delete_instance()


    def test_output_expiry_error_wo_state(self, rectest):
        """Test bad output of product expiry"""
        dis = ListeModel()
        dis.liste = "PERIME A SORTIR xxx"
        dis.fusion = "SORTIE DE PERIMES"
        dis.service = rectest["service"].code
        dis.type_servi = TypeServiListe.GlobaleBoite.value
        dis.num_ipp = PatientGlobal.Ipp.value
        dis.num_sej = PatientGlobal.Sejour.value
        dis.id_servi = 2
        dis.mode = TypeListe.Output.value
        dis.save()
        assert dis.pk > 0
        assert dis.etat == EtatListe.Vierge.value
        dis.delete_instance()

    def test_riedl_error_message_wo_liste(self, rectest):
        """Test create an output lot list"""
        with pytest.raises(MedianLibException) as e:
            err = ListeErrorModel()
            err.liste = "RIEDL LOT OUTPUT FAILED"
            err.mode = TypeListe.Output.value
            err.message = "Test message"
            err.save()
        assert "Original list not found !" in str(e.value)

    def test_riedl_create_output_lot(self, rectest):
        """Test create an output lot list"""
        dis = ListeModel()
        dis.liste = "RIEDL LOT OUTPUT"
        dis.fusion = "SORTIE DE LOTS"
        dis.service = rectest["service"].code
        dis.type_servi = TypeServiListe.RiedlBoite.value
        dis.num_ipp = PatientGlobal.Ipp.value
        dis.num_sej = PatientGlobal.Sejour.value
        dis.id_servi = 2
        dis.mode = TypeListe.Output.value
        dis.etat = EtatListe.Vierge.value
        dis.save()
        assert dis.pk > 0

        err = ListeErrorModel()
        err.liste = dis.liste
        err.mode = dis.mode
        err.message = "Test message"
        err.save()

        dis.delete_instance()

        # Check if error message is clean
        with pytest.raises(DoesNotExist):
            ListeErrorModel.get(liste=dis.liste, mode=dis.mode)

    def test_riedl_create_output_lot_check_state_undone(self, rectest):
        """Test create an output lot list
           When we add and error message, the state change to I
        """
        dis = ListeModel()
        dis.liste = "RIEDL LOT OUTPUT"
        dis.fusion = "SORTIE DE LOTS"
        dis.service = rectest["service"].code
        dis.type_servi = TypeServiListe.RiedlBoite.value
        dis.num_ipp = PatientGlobal.Ipp.value
        dis.num_sej = PatientGlobal.Sejour.value
        dis.id_servi = 2
        dis.mode = TypeListe.Output.value
        dis.etat = EtatListe.Vierge.value
        dis.save()
        assert dis.pk > 0

        err = ListeErrorModel()
        err.liste = dis.liste
        err.mode = dis.mode
        err.message = "Test message"
        err.save()

        lect = ListeModel.get(liste=dis.liste, mode=dis.mode)
        assert lect.etat == "V"
        assert lect.with_note == 1

        dis.delete_instance()

        # Check if error message is clean
        with pytest.raises(DoesNotExist):
            ListeErrorModel.get(liste=dis.liste, mode=dis.mode)

    def test_riedl_create_output_lot_check_state_delete_note(self, rectest):
        """Test create an output lot list
           When we add and error message, the state change to I
        """
        dis = ListeModel()
        dis.liste = "RIEDL LOT OUTPUT"
        dis.fusion = "SORTIE DE LOTS"
        dis.service = rectest["service"].code
        dis.type_servi = TypeServiListe.RiedlBoite.value
        dis.num_ipp = PatientGlobal.Ipp.value
        dis.num_sej = PatientGlobal.Sejour.value
        dis.id_servi = 2
        dis.mode = TypeListe.Output.value
        dis.etat = EtatListe.Vierge.value
        dis.save()
        assert dis.pk > 0

        err = ListeErrorModel()
        err.liste = dis.liste
        err.mode = dis.mode
        err.message = "Test message"
        err.save()

        lect = ListeModel.get(liste=dis.liste, mode=dis.mode)
        assert lect.etat == "V"
        assert lect.with_note == 1

        err.delete_instance()
        lect = ListeModel.get(liste=dis.liste, mode=dis.mode)
        assert lect.etat == "V"
        assert lect.with_note == 0

        dis.delete_instance()



    def test_riedl_create_output_lot_check_state_done(self, rectest):
        """Test create an output lot list
           When we add and error message, the state change to I
           except for state done "S"
        """
        dis = ListeModel()
        dis.liste = "RIEDL LOT OUTPUT"
        dis.fusion = "SORTIE DE LOTS"
        dis.service = rectest["service"].code
        dis.type_servi = TypeServiListe.RiedlBoite.value
        dis.num_ipp = PatientGlobal.Ipp.value
        dis.num_sej = PatientGlobal.Sejour.value
        dis.id_servi = 2
        dis.mode = TypeListe.Output.value
        dis.etat = EtatListe.Solde.value
        dis.save()
        assert dis.pk > 0

        err = ListeErrorModel()
        err.liste = dis.liste
        err.mode = dis.mode
        err.message = "Test message"
        err.save()

        lect = ListeModel.get(liste=dis.liste, mode=dis.mode)
        assert lect.etat == EtatListe.Solde.value

        dis.delete_instance()

        # Check if error message is clean
        with pytest.raises(DoesNotExist):
            ListeErrorModel.get(liste=dis.liste, mode=dis.mode)

    def test_create_work_item(self, rectest):
        """Create an workitem"""
        wi = WorkItemModel()
        wi.mode = TypeListe.Output.value
        wi.liste = 'RIEDL LOT OUTPUT'
        wi.item = '000001'
        wi.reference = "DEECALIBRE"
        wi.service = rectest["service"].code
        wi.quantite_dem = 2
        wi.interface = "35"
        wi.num_ipp = PatientGlobal.Ipp.value
        wi.num_sej = PatientGlobal.Sejour.value
        wi.save()
        assert wi.pk > 0
        wi.delete_instance()

    def test_deletion_list_item_workitem(self, rectest):
        """Delete liste and chack if item, workitem and error are removed"""
        dis = ListeModel()
        dis.liste = "RIEDL LOT OUTPUT"
        dis.fusion = "SORTIE DE LOTS"
        dis.service = rectest["service"].code
        dis.type_servi = TypeServiListe.RiedlBoite.value
        dis.num_ipp = PatientGlobal.Ipp.value
        dis.num_sej = PatientGlobal.Sejour.value
        dis.id_servi = 2
        dis.mode = TypeListe.Output.value
        dis.etat = EtatListe.Solde.value
        dis.interface = "35"
        dis.save()
        assert dis.pk > 0

        itm = ListeItemModel()
        itm.liste = dis.liste
        itm.mode = dis.mode
        itm.item = '000001'
        itm.reference = rectest["p1"].reference
        itm.dest = dis.service
        itm.qte_dem = 4
        itm.save()
        assert itm.pk > 0

        wi = WorkItemModel()
        wi.mode = TypeListe.Output.value
        wi.liste = dis.liste
        wi.item = itm.item
        wi.reference = itm.reference
        wi.service = itm.dest
        wi.quantite_dem = 2
        wi.interface = dis.interface
        wi.num_ipp = PatientGlobal.Ipp.value
        wi.num_sej = PatientGlobal.Sejour.value
        wi.save()
        assert wi.pk > 0

        wi2 = WorkItemModel()
        wi2.mode = TypeListe.Output.value
        wi2.liste = dis.liste
        wi2.item = itm.item
        wi2.reference = itm.reference
        wi2.service = itm.dest
        wi2.quantite_dem = 2
        wi2.interface = dis.interface
        wi2.num_ipp = PatientGlobal.Ipp.value
        wi2.num_sej = PatientGlobal.Sejour.value
        wi2.save()
        assert wi2.pk > 0

        err = ListeErrorModel()
        err.liste = dis.liste
        err.mode = dis.mode
        err.message = "Test message"
        err.save()

        lect = ListeModel.get(liste=dis.liste, mode=dis.mode)
        assert lect.etat == EtatListe.Solde.value

        # Delete Liste also delete Item, ListeError and WorkItem
        dis.delete_instance()

        with pytest.raises(DoesNotExist):
            ListeItemModel.get(liste=dis.liste, mode=dis.mode, item='000001')

        # Check if error message is clean
        with pytest.raises(DoesNotExist):
            ListeErrorModel.get(liste=dis.liste, mode=dis.mode)

        wkitms = WorkItemModel.select(WorkItemModel).where(
            WorkItemModel.liste == dis.liste, WorkItemModel.mode == dis.mode
        )

        assert len(wkitms) == 0

    def test_reset_picking_entry_list(self, rectest):

        dis = ListeModel()
        dis.liste = "sortie nominative 1"
        dis.fusion = "NOMINATIF"
        dis.service = rectest["service"].code
        dis.type_servi = TypeServiListe.Nominatif.value
        dis.num_ipp = "0002012033"
        dis.num_sej = "SEJ1234568"
        dis.id_servi = 2
        dis.mode = TypeListe.Input.value
        dis.etat = EtatListe.EnCours.value
        dis.interface = ""
        dis.save()
        assert dis.pk > 0

        with pytest.raises(ValueError):
            dis.reset_picking()

    def test_reset_picking_output_list(self, rectest):
        dis = ListeModel()
        dis.liste = "sortie nominative 1"
        dis.fusion = "NOMINATIF"
        dis.service = rectest["service"].code
        dis.type_servi = TypeServiListe.Nominatif.value
        dis.num_ipp = "0002012033"
        dis.num_sej = "SEJ1234568"
        dis.id_servi = 2
        dis.mode = TypeListe.Output.value
        dis.etat = EtatListe.EnCours.value
        dis.interface = ""
        dis.id_chargement = "956"
        dis.selectionne = 1
        dis.origine_sel = "ACCED_ST1"
        dis.valide_sel = True
        dis.id_plateau = "2"
        dis.save()
        assert dis.pk > 0

        itm = ListeItemModel()
        itm.liste = dis.liste
        itm.mode = dis.mode
        itm.item = '000001'
        itm.etat = EtatListe.EnCours.value
        itm.reference = rectest["p1"].reference
        itm.type_servi = dis.type_servi
        itm.dest = dis.service
        itm.num_ipp = dis.num_ipp
        itm.num_sej = dis.num_sej
        itm.qte_dem = 4
        itm.qte_serv = 2
        itm.save()
        assert itm.pk > 0

        wi = WorkItemModel()
        wi.mode = TypeListe.Output.value
        wi.liste = dis.liste
        wi.item = itm.item
        wi.reference = itm.reference
        wi.service = itm.dest
        wi.quantite_dem = 2
        wi.interface = dis.interface
        wi.num_ipp = dis.num_ipp
        wi.num_sej = dis.num_sej
        wi.save()
        assert wi.pk > 0

        wi2 = WorkItemModel()
        wi2.mode = TypeListe.Output.value
        wi2.liste = dis.liste
        wi2.item = itm.item
        wi2.reference = itm.reference
        wi2.service = itm.dest
        wi2.quantite_dem = 2
        wi2.interface = dis.interface
        wi2.num_ipp = dis.num_ipp
        wi2.num_sej = dis.num_sej
        wi2.save()
        assert wi2.pk > 0

        dis.reset_picking()

        res: ListeModel = ListeModel.get_by_id(dis.pk)

        assert res.etat == "V"
        assert res.id_chargement == ""
        assert res.id_plateau == ""
        assert res.id_peigne == dis.id_peigne
        assert res.selectionne == 0
        assert res.valide_sel == False

        res_item : ListeItemModel = ListeItemModel.get_by_id(itm.pk)

        assert res_item.etat == "V"
        assert res_item.qte_serv == 0

        f_workitems = WorkItemModel.select().where(
            WorkItemModel.liste == res.liste, WorkItemModel.mode == res.mode
        )
        assert len(f_workitems) == 0

