from datetime import datetime
import time
from peewee import DoesNotExist, fn

from median.constant import (
    EtatListe, PatientGlobal, TypeListe, TypeServiListe,
    EquipmentType, EcoType, MEDIANWEB_POSTE)
from median.database import mysql_db
from median.models import (
    Compteur, Endpoint, Magasin, Poste, Seuil,
    Product, Service, Stock, ListeModel, ListeItemModel)
from median.utils import compute_gtin_checksum, logger, get_counter

from ..base import BaseView, BaseViewException


class RiedlViewException(Exception):
    def __init__(self, message):
        self.message = message


class RiedlView(BaseView):
    """Represent a Riedl Instance"""

    poste = None
    warehouses = []
    endpoints = []
    num_inventory_line = 0

    def __init__(self, poste=None):
        """Implement an instance of the Riedl"""
        self.mandatory(poste, "poste must be required!")
        try:
            self.poste = Poste.get(poste=poste)
        except DoesNotExist:
            raise BaseViewException("Poste %s does not exists" % poste)

        # Retrieve all warehouse relate to this equipment (1 is necessary)
        self.warehouses = (
            Magasin()
            .select(Magasin)
            .where(Magasin.eco_type == EcoType.Riedl.value, Magasin.type_mag == poste)
        )

        self.endpoints = self.get_endpoints()

    def add_endpoint(self, name, label="Unknown", priority=3, ep_number=1):
        """Add new endpoints for the this Riedl
        Priority: 3 default, 0 more, 5 less

        """
        ap = Endpoint()
        ap.code = name
        ap.libelle = label
        ap.secteur = self.poste.poste
        ap.type_dest = "RIEDL"
        ap.ordre = priority
        ap.type_peigne = ep_number
        ap.save()

        self.endpoints = self.get_endpoints()
        return ap

    def get_endpoints(self):
        """Retrieve liste of all endpoints"""
        return (
            Endpoint()
            .select(Endpoint)
            .where(Endpoint.type_dest == "RIEDL", Endpoint.secteur == self.poste.poste)
        )

    def get_next_counter(self, counter_code):
        """
        Retrieve the value of a counter and return the value
        and increment the number for the next round
        """
        return self._get_next_code_counter("RIEDL_%s" % counter_code)

    def get_gtin_return_code(self):
        """
        Generate a GTIN code start with 233001
        """
        prefix = "233001"
        value = self._get_next_code_counter("GTIN_RETOUR")

        return compute_gtin_checksum("%s%06d" % (prefix, value), True)

    def get_gtin_taking_mode_code(self):
        """
        Generate a GTIN code start with 255001
        """
        prefix = "255001"
        value = self._get_next_code_counter("GTIN_PRELEVEMENT")

        return compute_gtin_checksum("%s%06d" % (prefix, value), True)

    def create_taking_mode(self, model_id=None):
        """
        Create an entry list for the taking mode, base on the unload list
        The quantity to return is deduce from the quantity attendee and quantity unload
        """
        if not model_id:
            raise BaseViewException("Id of the model is required!")

        #
        try:
            mId = ListeModel.get(pk=model_id)
            # if mId.etat != EtatListe.Solde.value:
            #     raise BaseViewException("List must be done: %d" % model_id)

        except DoesNotExist:
            raise BaseViewException("List not found: %d" % model_id)

        if mId.etat == EtatListe.Solde.value:
            new_list = self._taking_mode_done(mId)
        else:
            new_list = self._taking_mode_undone(mId)

        return new_list

    def generate_inventory(self, creator=MEDIANWEB_POSTE):
        """Generate and inventory, and return the id of the list"""
        warehouse = self.warehouses[0]
        lst = self._add_inventory_header(creator)

        # Retrieve the list of the product who have threashold to create items
        pro_rec = Product.select(Product.reference).join(
            Seuil, on=(Seuil.reference == Product.reference)
        ).where(Seuil.zone == warehouse.type_mag).order_by(Product.reference)

        for pro in pro_rec:
            self._add_inventory_line(lst, pro, warehouse)

        lst.nb_item = self.num_inventory_line
        lst.date_modification = datetime.now()
        lst.save()

        return (lst, self.num_inventory_line)

    def _add_inventory_header(self, creator=MEDIANWEB_POSTE) -> ListeModel:
        """
        Create an f_liste for the inventory
        """
        warehouse = self.warehouses[0]
        self.num_inventory_line = 0

        logger.debug("-- Riedl Inventory for %s / %s --" % (warehouse.mag, warehouse.type_mag))
        lst = ListeModel()
        lst.liste = "%s-%s" % (warehouse.mag, time.strftime("%Y%m%d-%H%M%S"))
        lst.mode = TypeListe.Inventory.value
        lst.etat = EtatListe.Vierge.value
        lst.service = 'TRAN'  # TODO: retrieve the key on f_config
        lst.type_servi = TypeServiListe.Inventory.value
        lst.id_servi = 8
        lst.fusion = 'INVENTAIRE'
        lst.nb_item = self.num_inventory_line
        lst.num_ipp = PatientGlobal.Ipp.value
        lst.num_sej = PatientGlobal.Sejour.value
        lst.zone_deb = EquipmentType.RIEDL.value
        lst.zone_fin = warehouse.type_mag
        lst.date_reception = datetime.now()
        lst.username = creator
        lst.save()
        return lst

    def _add_inventory_line(
        self, liste: ListeModel, product: Product, warehouse: Magasin
    ):
        """
        Add line to the current inventory
        """
        # For each product we retrieve all GTIN version 0
        gtin_version_zero = []
        # gtins = product.gtin_list()
        for g in product.gtin_list():
            if g.dossier == "0" and g.cip not in gtin_version_zero:
                gtin_version_zero.append(g.cip)

        for gtin in gtin_version_zero:
            # For each GTIN we create an item
            self.num_inventory_line += 1

            stk = Stock.select(
                fn.COUNT(Stock.pk).alias('boites'),
                fn.IFNULL(fn.SUM(Stock.quantite), 0).alias('total')
            ).where(
                Stock.magasin == warehouse.mag,
                Stock.reference == product.reference,
                Stock.cip == gtin
            )

            logger.debug("item new %s" % gtin)
            itm = ListeItemModel()
            itm.mode = liste.mode
            itm.liste = liste.liste
            itm.item = '%06d' % self.num_inventory_line
            itm.dest = liste.service
            itm.magasin = warehouse.mag
            itm.qte_dem = stk[0].total
            itm.qte_serv = 0
            itm.quantite_disp = stk[0].boites
            itm.num_ipp = liste.num_ipp
            itm.num_sej = liste.num_sej
            itm.reference = product.reference
            itm.user = liste.username
            itm.lot = ""
            itm.tperemp = ""
            itm.code_liv = gtin
            itm.contenant = gtin
            itm.type_servi = liste.type_servi
            itm.id_chargement = get_counter('RIEDL_INVENTORY')
            itm.cip = gtin
            itm.save()

    def _taking_mode_undone(self, model_id=None):
        """
        Create a taking mode list from a new list
        Compute box quantity per
        """
        logger.info("create taking mode list")
        item_list = []

        # Retrieve the taking mode endpoint for this endpoint
        try:
            endp = Endpoint.get(type_dest='RIEDL', secteur=model_id.zone_fin, type_peigne=model_id.no_pilulier)
            logger.info("Exit unload standard: %s" % endp.code)
            if endp.adr1:
                # If endpoint taking mode not defined, we cannot create the list
                # We can creat an f_liste_error, to said not taking mode endpoint defined !
                logger.error("Exit unload not found for %s" % endp.code)
                return None
            endp_prelev = Endpoint.get(type_dest='RIEDL', code=endp.adr1)
            logger.info("Exit unload found for: %s" % endp_prelev.code)
        except DoesNotExist:
            # TODO: if endpoint not exists, we quit or raise and error
            return None

        lst2 = ListeModel()
        lst2.mode = model_id.mode
        lst2.liste = model_id.liste + "-P"
        lst2.liste_origine = model_id.liste
        lst2.valide_sel = False
        lst2.etat = EtatListe.Vierge.value
        lst2.type_servi = TypeServiListe.GlobaleBoite.value
        lst2.fusion = "PRELEVEMENT"
        lst2.service = model_id.service
        lst2.num_sej = model_id.num_sej
        lst2.ipp = model_id.ipp
        lst2.num_ipp = PatientGlobal.Ipp.value
        lst2.num_sej = PatientGlobal.Sejour.value
        lst2.zone_deb = model_id.zone_deb
        lst2.zone_fin = model_id.zone_fin
        lst2.pos_pilulier = 1
        lst2.no_pilulier = endp_prelev.type_peigne
        lst2.nb_item = 1
        lst2.selectionne = 1

        # Retrieve the warehouse for this riedl
        try:
            mag = Magasin.get(Magasin.type_mag == model_id.zone_fin, Magasin.eco_type == 'L')
        except DoesNotExist:
            # TODO: THis case is not possible
            pass

        # For each line, compute line with stock and check if necessary to split the list
        itms = ListeItemModel.select(ListeItemModel).where(
            ListeItemModel.liste == model_id.liste, ListeItemModel.mode == model_id.etat
        )
        line = 1
        for i in itms:
            cpt_qte = 0
            # old_qte = 0
            # check quantity to unload (only unblock stock line)
            stk = Stock.select(Stock).where(Stock.magasin == mag, Stock.reference == i.reference,
                                            Stock.bloque == 0).order_by(Stock.date_peremption)
            for s in stk:
                cpt_qte += s.quantite

                if i.qte_dem == cpt_qte:
                    # If quantity is the same, we move to the next item
                    break
                elif i.qte_dem > cpt_qte:
                    # Not enougth quantity we continue with another stock.
                    continue
                else:
                    new_item = ListeItemModel()
                    new_item.liste = lst2.liste
                    new_item.mode = lst2.mode
                    new_item.etat = EtatListe.Vierge.value
                    new_item.dest = lst2.service
                    new_item.fraction = 100
                    new_item.type_servi = lst2.type_servi
                    new_item.reference = i.reference
                    new_item.item = str(line).zfill(6)
                    new_item.qte_dem = cpt_qte - i.qte_dem      # Quantité a retournée.
                    new_item.qte_serv = 0
                    new_item.type_servi = lst2.type_servi
                    new_item.dtprise = datetime.now()
                    new_item.num_ipp = lst2.num_ipp
                    new_item.num_sej = lst2.num_sej
                    new_item.cip = s.cip
                    new_item.ucd = s.ucd

                    item_list.append(new_item)

                    line += 1

                    # TODO: after we remove the quantity on the original list
                    i.qte_serv = cpt_qte - i.qte_dem        # on indique la quantité retournéé comme deja prise
                    i.save()
        # If items on the list, wee
        if item_list:
            lst2.nb_item = line
            lst2.save()
            for itm in item_list:
                itm.save()

            return lst2

        return None

    def _taking_mode_done(self, model_id=None):
        """
        Create a taking mode list from a unload list done
        """
        tk_obj = ListeModel()
        tk_obj.mode = TypeListe.Input.value
        tk_obj.etat = EtatListe.Vierge.value
        tk_obj.liste = "%s-P" % model_id.liste
        tk_obj.zone_deb = model_id.zone_deb
        tk_obj.zone_fin = model_id.zone_fin
        tk_obj.fusion = "PRELEVEMENT"
        tk_obj.type_servi = TypeServiListe.Prelevement.value
        tk_obj.service = model_id.service
        tk_obj.nb_item = 0
        tk_obj.save()

        line_count = 0
        for itm in ListeItemModel.select(ListeItemModel).where(
            ListeItemModel.mode == model_id.mode, ListeItemModel.liste == model_id.liste
        ):

            if itm.qte_dem < itm.qte_serv:
                line_count += 1
                li_obj = ListeItemModel()
                li_obj.mode = tk_obj.mode
                li_obj.etat = EtatListe.Vierge.value
                li_obj.liste = tk_obj.liste
                li_obj.type_servi = tk_obj.type_servi
                li_obj.reference = itm.reference
                li_obj.qte_dem = itm.qte_serv - itm.qte_dem
                li_obj.qte_serv = 0
                li_obj.item = "%06d" % line_count
                li_obj.lot = itm.lot  # Use use the last box batch lot
                li_obj.tperemp = itm.tperemp  # Use use the last box expiry date
                li_obj.contenant = self.get_gtin_taking_mode_code()
                li_obj.info = itm.info  # Store the UCD for printing
                li_obj.id_pilulier = itm.id_pilulier  # Print the original gtin
                li_obj.cip = itm.cip
                li_obj.ucd = itm.ucd
                li_obj.save()

        tk_obj.nb_item = line_count
        tk_obj.liste_origine = model_id.liste
        tk_obj.save()

        # Update model_id with new list generate
        model_id.id_peigne = 1
        model_id.save()

        return tk_obj

    def create_ward_return(
        self, ward_code, reference, quantity=0.0, batch=None, expiry_date=None, ucd=""
    ):
        """
        Create an entry list for a ward return
        We concatenate list if on the same ward and in draft
        """
        # Check if the ward exists
        try:
            Service.get(code=ward_code)
        except DoesNotExist:
            raise BaseViewException("Ward %s does not exists" % ward_code)

        # Check if the Drugs exists
        try:
            Product.get(reference=reference)
        except DoesNotExist:
            raise BaseViewException("Reference %s does not exists" % ward_code)

        if quantity == 0.0:
            raise BaseViewException("Quantity must be greather than 0")

        # Search a return list for this ward
        try:
            lst = ListeModel.get(
                service=ward_code,
                mode=TypeListe.Input.value,
                etat=EtatListe.Vierge.value,
                type_servi=TypeServiListe.Retour.value,
                zone_deb="RIEDL",
                zone_fin=self.poste.poste,
            )
            lst.nb_item = lst.nb_item + 1
            lst.save()
        except DoesNotExist:
            # Retrieve the next counter value
            counter = self.get_next_counter("RETOUR")
            logger.info("Next return counter %s" % counter)

            lst = ListeModel()
            lst.mode = TypeListe.Input.value
            lst.etat = EtatListe.Vierge.value
            lst.liste = "%s-%s-%06d" % (self.warehouses[0].mag, ward_code, counter)
            lst.type_servi = TypeServiListe.Retour.value
            lst.nb_item = 1
            lst.fusion = "RETOUR"
            lst.service = ward_code
            lst.zone_deb = "RIEDL"
            lst.zone_fin = self.poste.poste
            lst.sous_secteur = self.poste.poste
            lst.num_ipp = PatientGlobal.Ipp.value
            lst.num_sej = PatientGlobal.Sejour.value
            lst.save()

        itm = ListeItemModel()
        itm.liste = lst.liste
        itm.mode = lst.mode
        itm.etat = EtatListe.Vierge.value
        itm.item = "%06d" % lst.nb_item
        itm.reference = reference
        itm.qte_dem = quantity
        itm.type_servi = lst.type_servi
        itm.dest = lst.service
        itm.sous_secteur = lst.sous_secteur
        itm.lot = batch.upper()
        itm.num_ipp = lst.num_ipp
        itm.num_sej = lst.num_sej

        if expiry_date:
            itm.tperemp = expiry_date

        # Create a CIP to print
        itm.contenant = self.get_gtin_return_code()
        itm.info = ucd

        itm.save()

        return lst

    def _get_next_code_counter(self, counter_code):
        """
        Retrieve the next counter, and increment it
        If counter not exists, we create it
        """
        result = 1
        try:
            with mysql_db.atomic():
                cpt = (
                    Compteur.select()
                    .where(Compteur.cle == counter_code)
                    .for_update()
                    .get()
                )
                result = cpt.val
                cpt.val = cpt.val + 1
                cpt.save()

        except DoesNotExist:
            cpt = Compteur()
            cpt.cle = counter_code
            cpt.val = 1
            cpt.save()
        return result

    def _compute_checksum(self, barcode):
        """
        Compute the checksum of the barcode
        """
        if len(barcode) > 12:
            barcode = barcode[:12]
        return compute_gtin_checksum(barcode)


Riedl = RiedlView
