from datetime import datetime
import time
from peewee import DoesNotExist, fn

from median.constant import (
    EtatListe, PatientGlobal, TypeListe, TypeServiListe,
    EquipmentType, EcoType, MEDIANWEB_POSTE)
from median.models import (
    Magasin, Poste, Seuil, Product,
    Stock, ListeModel, ListeItemModel)
from median.utils import logger, get_counter

from ..base import BaseView, BaseViewException


class ExternalViewException(Exception):
    def __init__(self, message):
        self.message = message


class ExternalView(BaseView):
    """Represent a External storage Instance"""

    poste = None
    warehouses = []
    num_inventory_line = 0

    def __init__(self, poste=None):
        """Implement an instance of the External storage"""
        self.mandatory(poste, "poste must be required!")
        try:
            self.poste = Poste.get(poste=poste)
        except DoesNotExist:
            raise BaseViewException("Poste %s does not exists" % poste)

        # Retrieve all warehouse relate to this equipment (1 is necessary)
        self.warehouses = (
            Magasin()
            .select(Magasin)
            .where(Magasin.eco_type == EcoType.Riedl.value, Magasin.type_mag == poste)
        )

    def generate_inventory(self, creator=MEDIANWEB_POSTE):
        """Generate and inventory, and return the id of the list"""
        warehouse = self.warehouses[0]
        lst = self._add_inventory_header(creator)

        # Retrieve the list of the product who have threashold to create items
        pro_rec = Product.select(
            Product.reference, (Seuil.fraction).alias("fraction")
        ).join(
            Seuil, on=(Seuil.reference == Product.reference)
        ).where(Seuil.zone == warehouse.type_mag).order_by(Product.reference)

        for rec in pro_rec:
            self._add_inventory_line(lst, rec, warehouse)

        lst.nb_item = self.num_inventory_line
        lst.date_modification = datetime.now()
        lst.save()

        return (lst, self.num_inventory_line)

    def _add_inventory_header(self, creator=MEDIANWEB_POSTE) -> ListeModel:
        """
        Create an f_liste for the inventory
        """
        warehouse = self.warehouses[0]
        self.num_inventory_line = 0

        logger.debug("-- External Inventory for %s / %s --" % (warehouse.mag, warehouse.type_mag))
        lst = ListeModel()
        lst.liste = "%s-%s" % (warehouse.mag, time.strftime("%Y%m%d-%H%M%S"))
        lst.mode = TypeListe.Inventory.value
        lst.etat = EtatListe.Vierge.value
        lst.service = 'TRAN'  # TODO: retrieve the key on f_config
        lst.type_servi = TypeServiListe.Inventory.value
        lst.id_servi = 8
        lst.fusion = 'INVENTAIRE'
        lst.nb_item = self.num_inventory_line
        lst.num_ipp = PatientGlobal.Ipp.value
        lst.num_sej = PatientGlobal.Sejour.value
        lst.zone_deb = EquipmentType.EXTERNE.value
        lst.zone_fin = warehouse.type_mag
        lst.date_reception = datetime.now()
        lst.username = creator
        lst.save()
        return lst

    def _add_inventory_line(
        self, liste: ListeModel, product, warehouse: Magasin
    ):
        """
        Add line to the current inventory
        """
        self.num_inventory_line += 1

        stk = Stock.select(
            fn.COUNT(Stock.pk).alias('boites'),
            fn.IFNULL(fn.SUM(Stock.quantite), 0).alias('total')
        ).where(
            Stock.magasin == warehouse.mag,
            Stock.reference == product.reference,
            Stock.cip == product.seuil.fraction
        )

        logger.debug("item new %s" % product.reference)
        itm = ListeItemModel()
        itm.mode = liste.mode
        itm.liste = liste.liste
        itm.item = '%06d' % self.num_inventory_line
        itm.dest = liste.service
        itm.magasin = warehouse.mag
        itm.qte_dem = stk[0].total
        itm.qte_serv = 0
        itm.quantite_disp = stk[0].boites
        itm.fraction = product.seuil.fraction
        itm.num_ipp = liste.num_ipp
        itm.num_sej = liste.num_sej
        itm.reference = product.reference
        itm.user = liste.username
        itm.lot = ""
        itm.tperemp = ""
        itm.code_liv = ""
        itm.contenant = ""
        itm.type_servi = liste.type_servi
        itm.id_chargement = get_counter('EXTERNAL_INVENTORY')
        itm.cip = ""
        itm.save()
