# flake8: noqa
import pytest
from median.utils import compute_gtin_checksum, get_counter, padding_address, date_add_month
from median.base import MedianLibException
from median.models import Compteur
import datetime


class TestUtils:
    def test_compute_bad_length_checksum(self):
        with pytest.raises(MedianLibException) as e:
            compute_gtin_checksum("23300100000")
        assert "GTIN must have 12 characters!" in str(e.value)

    def test_compute_empty_gtin_checksum(self):
        with pytest.raises(MedianLibException) as e:
            compute_gtin_checksum("")
        assert "GTIN cannot be null" in str(e.value)

    def test_compute_good_checksum(self):
        chk = compute_gtin_checksum("233001000000")
        assert chk == "3"

    def test_compute_good_checksum_full(self):
        chk = compute_gtin_checksum("233001000000", True)
        assert chk == "2330010000003"

    def test_get_unknown_counter(self):
        with pytest.raises(MedianLibException):
            get_counter('PYTEST_UNK')

    def test_get_unknown_counter_and_create(self):
        res = get_counter('PYTEST_UNK_CREA', force_create=True)
        assert res == 1
        cr = Compteur.get(cle='PYTEST_UNK_CREA')
        assert cr.val == 2
        cr.delete_instance()

    def test_get_existing_counter(self):
        cpt = Compteur()
        cpt.cle = 'PYTEST'
        cpt.val = 2
        cpt.save()

        res = get_counter('PYTEST')
        assert res == 2

        # check if the next value is correct
        cr = Compteur.get(cle='PYTEST')
        assert cr.val == 3
        cr.delete_instance()

    def test_get_existing_counter_increment_five(self):
        cpt = Compteur()
        cpt.cle = 'PYTEST5'
        cpt.val = 2
        cpt.save()

        res = get_counter('PYTEST5', 5)
        assert res == 2

        # check if the next value is correct
        cr = Compteur.get(cle='PYTEST5')
        assert cr.val == 7
        cr.delete_instance()

    def test_padding_string(self):
        assert padding_address('ST1') == "ST1"

    def test_padding_list(self):
        assert padding_address(['ST1']) == "ST1"

    def test_padding_list_2_integer(self):
        assert padding_address(['ST1', 1]) == "ST1.  1"

    def test_padding_string_2(self):
        assert padding_address('ST1.1') == "ST1.  1"

    def test_padding_list_2(self):
        assert padding_address(['ST1', '1']) == "ST1.  1"

    def test_padding_string_3(self):
        assert padding_address('ST1.1.2') == "ST1.  1.  2"

    def test_padding_list_3(self):
        assert padding_address(['ST1', '1', 2]) == "ST1.  1.  2"

    def test_padding_string_4(self):
        assert padding_address('ST1.1.2.3') == "ST1.  1.  2.  3"

    def test_padding_list_4(self):
        assert padding_address(['ST1', '1', 2, 3]) == "ST1.  1.  2.  3"

    def test_padding_string_5(self):
        assert padding_address('ST1.1. 2.3. 5') == "ST1.  1.  2.  3.  5"

    def test_padding_list_5(self):
        assert padding_address(['ST1', '1', 2, 6, '3']) == "ST1.  1.  2.  6.  3"

    def test_padding_string_bad_space(self):
        assert padding_address('ST1 .1 . 2 .3 . 5 ') == "ST1.  1.  2.  3.  5"

    def test_padding_string_search(self):
        assert padding_address('ST1 .1 . 2 . %') == "ST1.  1.  2.%"

    def test_padding_string_search_2(self):
        assert padding_address('ST1 .1 . % . %') == "ST1.  1.%.%"

    def test_add_10_month(self):
        dt = datetime.date(2024, 10, 20)
        assert date_add_month(dt, 10).strftime("%Y%m%d") == datetime.date(2025, 8, 20).strftime("%Y%m%d")

    def test_add_1_month(self):
        dt = datetime.date(2024, 12, 1)
        assert date_add_month(dt, 1).strftime("%Y%m%d") == datetime.date(2025, 1, 1).strftime("%Y%m%d")
