# flake8: noqa
import pytest
from peewee import DoesNotExist
from median.base import BaseViewException, BaseViewDuplicate
from median.models import ListeModel, ListeItemModel, Gpao, Historique, Product, Seuil, Magasin, Config, Service
from median.views import SeuilView


@pytest.fixture()
def rectest():
    print("Running method level setUp")
    rec_data = {}
    yield rec_data

    print("Running method level tearDown")
    # Remove all refill list afeter the test
    refill = ListeModel.select().where(ListeModel.mode == "E", ListeModel.fusion == "REASSORT")
    for rf in refill:
        rf.delete_instance()

    # Remove all GPAO product
    refgp = Gpao.select().where(Gpao.type_mvt == "R")
    for rgp in refgp:
        rgp.delete_instance()

    # Remove all histories reference
    hist = Historique.select().where(Historique.type_mouvement == "REF")
    for h in hist:
        h.delete_instance()


class TestObjectSeuil:

    def setup_method(self, method):
        Seuil.delete().execute()
        Service.delete().execute()
        Gpao.delete().execute()
        Historique.delete().execute()
        ListeModel.delete().execute()
        ListeItemModel.delete().execute()
        Magasin.get_or_create(mag="ST1", type_mag="CUEILLETTE_1", defaults={
            "eco_type": "C",
            "type_machine": "ACCED_220",
            "id_zone": 1000,
            "id_robot": 2000
        })
        Magasin.get_or_create(mag="ST2", type_mag="CUEILLETTE_2", defaults={
            "eco_type": "C",
            "type_machine": "ACCED_220",
            "id_zone": 1000,
            "id_robot": 2000
        })
        Magasin.get_or_create(mag="ST3", type_mag="CUEILLETTE_3", defaults={
            "eco_type": "C",
            "type_machine": "ACCED_120",
            "id_zone": 1000,
            "id_robot": 2000
        })
        Product.get_or_create(reference="DEVERMINAGE", defaults={
            "designation": "deverminage"
        })
        Product.get_or_create(reference="DOLIPRANE", defaults={
            "designation": "doliprane"
        })
        Seuil.get_or_create(reference="DOLIPRANE", zone="CUEILLETTE_1", fraction=100, defaults={
            "stock_mini": 100,
            "stock_maxi": 200,
        })
        Seuil.get_or_create(reference="DEVERMINAGE", zone="CUEILLETTE_1", fraction=100, defaults={
            "stock_mini": 100,
            "stock_maxi": 200,
        })
        Seuil.get_or_create(reference="DEVERMINAGE", zone="CUEILLETTE_1", fraction=50, defaults={
            "stock_mini": 100,
            "stock_maxi": 200,
        })
        Seuil.get_or_create(reference="DEVERMINAGE", zone="CUEILLETTE_2", fraction=100, defaults={
            "stock_mini": 100,
            "stock_maxi": 200,
        })
        Seuil.get_or_create(reference="DEVERMINAGE", zone="CUEILLETTE_2", fraction=50, defaults={
            "stock_mini": 100,
            "stock_maxi": 200,
        })
        Seuil.get_or_create(reference="DEVERMINAGE", zone="CUEILLETTE_2", fraction=25, defaults={
            "stock_mini": 100,
            "stock_maxi": 200,
        })
        Seuil.get_or_create(reference="DEVERMINAGE", zone="CUEILLETTE_3", fraction=100, defaults={
            "stock_mini": 100,
            "stock_maxi": 200,
        })
        Service.get_or_create(code="TRAN", defaults={
            "libelle": "Transfert",
            "type_dest": "SERVICE",
        })
        Config.get_or_create(poste="TOUS", cle="cfg", propriete="k_ua_transfert", defaults={
            "type_value": "C",
            "value": "TRAN",
        })

    def teardown_method(self, method):
        Product.delete().where(Product.reference=="DEVERMINAGE").execute()
        Seuil.delete().execute()
        Magasin.delete().where(Magasin.mag=="ST1").execute()
        Magasin.delete().where(Magasin.mag=="ST2").execute()
        Magasin.delete().where(Magasin.mag=="ST3").execute()

    def test_list_wo_product_wo_mag(self):
        """Display list without product or magasin"""
        with pytest.raises(BaseViewException):
            s = SeuilView()
            s.liste()

    def test_list_w_product_unknown_wo_mag(self):
        """Check with unknown product"""
        with pytest.raises(BaseViewException):
            s = SeuilView()
            s.liste(produit="INCONNU")

    def test_list_wo_product_w_mag_unknown(self):
        """Check with unknown magasin"""
        with pytest.raises(BaseViewException):
            s = SeuilView()
            s.liste(magasin="INCONNU")

    def test_list_w_product_deveminage_wo_mag(self):
        """Check with DEVERMINAGE"""
        s = SeuilView()
        seuil_list = s.liste(produit="DEVERMINAGE")
        assert len(seuil_list) == 6

    def test_list_w_product_deveminage_w_mag_st1(self):
        """Check with DEVERMINAGE and mag ST1"""
        s = SeuilView()
        seuil_list = s.liste(produit="DEVERMINAGE", magasin="ST1")
        assert len(seuil_list) == 2

    def test_list_w_product_deveminage_w_mag_st2(self):
        """Check with DEVERMINAGE and mag ST2"""
        s = SeuilView()
        seuil_list = s.liste(produit="DEVERMINAGE", magasin="ST2")
        assert len(seuil_list) == 3

    def test_list_wo_product_w_mag_st1(self):
        """Check with mag ST2"""
        s = SeuilView()
        seuil_list = s.liste(magasin="ST1")
        assert len(seuil_list) == 3

    def test_list_wo_product_w_mag_st2(self):
        """Check with mag ST2"""
        s = SeuilView()
        seuil_list = s.liste(magasin="ST2")
        assert len(seuil_list) == 3
        for ls in seuil_list:
            assert ls["pk"] > 0

    def test_reappro_unknown_product(self):
        """Raise an error if product is unknown"""
        with pytest.raises(BaseViewException):
            s = SeuilView()
            s.reappro(produit="INCONNU", quantite=10)

    def test_reappro_wo_product_wo_magasin(self):
        """Raise an error if no product or no magasin"""
        with pytest.raises(BaseViewException):
            s = SeuilView()
            s.reappro(quantite=10)

    def test_reappro_w_product_wo_magasin(self):
        """Raise an error if no magasin"""
        with pytest.raises(BaseViewException) as e:
            s = SeuilView()
            s.reappro(produit="DEVERMINAGE", quantite=10)

        assert "Le magasin est obligatoire" in str(e.value)

    def test_reappro_wo_product_w_magasin(self):
        """Raise an error if no product"""
        with pytest.raises(BaseViewException) as e:
            s = SeuilView()
            s.reappro(magasin="ST1", quantite=10)
        assert "Le produit est obligatoire" in str(e.value)

    def test_reappro_bad_magasin(self):
        """Raise an error if magasin is unknwown"""
        with pytest.raises(BaseViewException) as e:
            s = SeuilView()
            s.reappro(produit="DEVERMINAGE", magasin="ST66", quantite=10)
        assert "Le magasin n'existe pas dans la base" in str(e.value)

    def test_reappro_zero_quantity(self):
        """Raise an error if empty quantity"""
        with pytest.raises(BaseViewException) as e:
            s = SeuilView()
            s.reappro(produit="DEVERMINAGE", magasin="ST1", quantite=0)
        assert "La quantité doit être supérieure à zéro" in str(e.value)

    def test_reappro_unknown_product(self):
        """Raise an error if product is unknown"""
        with pytest.raises(BaseViewException) as e:
            s = SeuilView()
            s.reappro(produit="INCONNU", magasin="ST1", quantite=10)
        assert "Le produit n'existe pas dans la base" in str(e.value)

    def test_reappro_deverminage_st1(self, rectest):
        """Add deverminage to ST1 quantity 10"""
        s = SeuilView()
        res = s.reappro(produit="DEVERMINAGE", magasin="ST1", quantite=10)
        assert "CUEILLETTE_1 Unitaire" in res
        # vérification f_gpao
        gp = Gpao.select().where(Gpao.type_mvt == "R")
        assert len(gp) == 1
        for g in gp:
            g.delete_instance()
        # suppression de la liste et de ces items
        dl = ListeModel.get(liste=res)
        assert dl.nb_item == 1
        dl.delete_instance()
        his = Historique.select().where(Historique.type_mouvement == "REF")
        assert len(his) == 1

    def test_reappro_deverminage_dolip_st1(self, rectest):
        """Add deverminage and doliprane to ST1 quantity 10"""
        s = SeuilView()
        res = s.reappro(produit="DEVERMINAGE", magasin="ST1", quantite=10)
        assert "CUEILLETTE_1 Unitaire" in res
        res = s.reappro(produit="DOLIPRANE", magasin="ST1", quantite=8)
        assert "CUEILLETTE_1 Unitaire" in res
        # vérification f_gpao
        gp = Gpao.select().where(Gpao.type_mvt == "R")
        assert len(gp) == 2
        for g in gp:
            g.delete_instance()
        # suppression de la liste et de ces items
        dl = ListeModel.get(liste=res)
        assert dl.nb_item == 2
        dl.delete_instance()
        his = Historique.select().where(Historique.type_mouvement == "REF")
        assert len(his) == 2

    def test_reappro_deverminage_f50_st1(self, rectest):
        """Add deverminage to ST1 quantity 10 with fraction 50"""
        s = SeuilView()
        res = s.reappro(produit="DEVERMINAGE", magasin="ST1", quantite=10, fraction=50)
        assert "CUEILLETTE_1 Unitaire" in res
        # vérification f_gpao
        gp = Gpao.select().where(Gpao.type_mvt == "R")
        assert len(gp) == 1
        assert gp[0].qte == 10
        assert gp[0].fraction == 50
        for g in gp:
            g.delete_instance()
        # suppression de la liste et de ces items
        dl = ListeModel.get(liste=res)
        assert dl.nb_item == 1
        dl.delete_instance()
        his = Historique.select().where(Historique.type_mouvement == "REF")
        assert len(his) == 1

    def test_reappro_deverminage_f25_st1(self, rectest):
        """Add deverminage to ST1 quantity 20 with fraction 25"""
        s = SeuilView()
        res = s.reappro(produit="DEVERMINAGE", magasin="ST1", quantite=20, fraction=25)
        assert "CUEILLETTE_1 Unitaire" in res
        # vérification f_gpao
        gp = Gpao.select().where(Gpao.type_mvt == "R")
        assert len(gp) == 1
        assert gp[0].qte == 20
        assert gp[0].fraction == 25
        for g in gp:
            g.delete_instance()
        # suppression de la liste et de ces items
        dl = ListeModel.get(liste=res)
        assert dl.nb_item == 1
        dl.delete_instance()
        his = Historique.select().where(Historique.type_mouvement == "REF")
        assert len(his) == 1

    def test_reappro_calcul_f25_st1(self, rectest):
        """Add deverminage to ST1 quantity 20 with fraction 25"""
        s = SeuilView()
        res = s.reappro_calcul("CUEILLETTE_1")
        assert len(res) == 3

    def test_reappro_calcul_f25_st1_ignore_min_threshold(self, rectest):
        s = SeuilView()
        res = s.reappro_calcul(type_magasin="CUEILLETTE_1", ignore_min_threshold=True)
        assert len(res) == 3
