# flake8: noqa
import pytest
from peewee import DoesNotExist
from median.base import BaseViewException, BaseViewDuplicate
from median.models import Magasin, Adresse, Stock
from median.views import EmplacementView, AdresseView
from median.utils import padding_address


class TestObjectAdresse:
    def test_init_adresse_sans_magasin(self):
        """Initialise une adresse sans magasin, erreur de création"""
        with pytest.raises(BaseViewException):
            emp = AdresseView("ST1.  1.  1.  1.  1")
            emp.creer()

    def test_init_adresse_avec_magasin(self):
        """Creation de l'adresse"""
        emp = AdresseView("ST1.  1.  1.  1.  1", "ST1")
        emp.creer()
        assert emp.id > 0

    def test_init_adresse_en_doublons(self):
        """L'emplacement est en doublons, nous ne pouvons pas le réecréer"""
        with pytest.raises(BaseViewException):
            emp2 = AdresseView("ST1.  1.  1.  1.  1", "ST1")
            emp2.creer()

    def test_init_emplacement_fail(self):
        """Test création emplacement qui doit retourner une erreur"""
        with pytest.raises(BaseViewException):
            emp = EmplacementView()
            emp.creer()

    def test_supprime_adresse(self):
        """Suppression de l'emplacement créé pour les tests"""
        emp = AdresseView("ST1.  1.  1.  1.  1")
        assert emp.supprimer()

    def test_supprime_adresse_inexistante(self):
        """Suppression de l'emplacement qui n'existe pas"""
        with pytest.raises(BaseViewException):
            emp = AdresseView("ST9.  0.  0.  0.  0")
            emp.supprimer()


class TestObjectEmplacement:

    def setup_method(self, method):
        Adresse.delete().execute()
        Magasin.get_or_create(mag="ST1", type_mag="CUEILLETTE_1", defaults={
            "eco_type": "C",
            "type_machine": "ACCED_220",
            "id_zone": 1000,
            "id_robot": 2000
        })
        Magasin.get_or_create(mag="ST2", type_mag="CUEILLETTE_2", defaults={
            "eco_type": "C",
            "type_machine": "ACCED_220",
            "id_zone": 1000,
            "id_robot": 2000
        })
        Magasin.get_or_create(mag="ST3", type_mag="CUEILLETTE_3", defaults={
            "eco_type": "C",
            "type_machine": "ACCED_120",
            "id_zone": 1000,
            "id_robot": 2000
        })
        Magasin.get_or_create(mag="ST4", type_mag="ACCED_ST4", defaults={
            "eco_type": "C",
            "type_machine": "ACCED_V2",
            "id_zone": 1000,
            "id_robot": 2000
        })
        Magasin.get_or_create(mag="ST5", type_mag="ACCED_ST5", defaults={
            "eco_type": "C",
            "type_machine": "ACCED_V3",
            "id_zone": 1000,
            "id_robot": 2000
        })
        Magasin.get_or_create(mag="U01", type_mag="ASTUS_1", defaults={
            "eco_type": "A",
            "type_machine": "ASTUS_V3",
            "id_zone": 1500,
            "id_robot": 2500
        })
        Magasin.get_or_create(mag="DC1", type_mag="COUPE_10", defaults={
            "eco_type": "T",
            "type_machine": "MC_500",
            "id_zone": 1600,
            "id_robot": 2600
        })
        # Create 10 free addresses
        for i in range(14):
            tmp_adr = ['ST1', 1, i, 1, 1]
            Adresse.get_or_create(adresse=padding_address(tmp_adr), defaults={
                "etat": "L",
                "format": "BOITE PASS",
                "magasin": tmp_adr[0],
                "contenant": "",
                "bloque": 0,
            })
            tmp_adr = ['ST1', 1, i, 1, 3]
            Adresse.get_or_create(adresse=padding_address(tmp_adr), defaults={
                "etat": "L",
                "format": "BOITE PASS",
                "magasin": tmp_adr[0],
                "contenant": "",
                "bloque": 0,
            })
        for i in range(14):  # Magasin astus
            tmp_adr = ['U01', 'A', i, 1]
            Adresse.get_or_create(adresse=padding_address(tmp_adr), defaults={
                "etat": "L",
                "format": "BAC 1/7 B",
                "magasin": tmp_adr[0],
                "contenant": "bP1/70000%s1" % str(i),
                "bloque": 0,
            })
            tmp_adr = ['U01', 'A', i, 2]
            Adresse.get_or_create(adresse=padding_address(tmp_adr), defaults={
                "etat": "L",
                "format": "BAC 1/7 B",
                "magasin": tmp_adr[0],
                "contenant": "bP1/70000%s7" % str(i),
                "bloque": 0,
            })
            tmp_adr = ['U01', 'M', i, 1]
            Adresse.get_or_create(adresse=padding_address(tmp_adr), defaults={
                "etat": "L",
                "format": "BAC 1/7 H",
                "magasin": tmp_adr[0],
                "contenant": "hP1/70000%s1" % str(i),
                "bloque": 0,
            })
            tmp_adr = ['U01', 'M', i, 2]
            Adresse.get_or_create(adresse=padding_address(tmp_adr), defaults={
                "etat": "L",
                "format": "BAC 1/7 H",
                "magasin": tmp_adr[0],
                "contenant": "hP1/70000%s1" % str(i),
                "bloque": 0,
            })
        for i in range(6):
            tmp_adr = ['DC1', i, 1]
            Adresse.get_or_create(adresse=padding_address(tmp_adr), defaults={
                "etat": "L",
                "format": "BOITE PASS",
                "magasin": tmp_adr[0],
                "contenant": "100000%i" % i,
                "bloque": 0,
            })
        tmp_adr = ['ST1', 2, 1, 1]
        Adresse.get_or_create(adresse=padding_address(tmp_adr), defaults={
            "etat": "O",
            "format": "BOITE PASS",
            "magasin": tmp_adr[0],
            "contenant": "123456789",
            "bloque": 0,
        })
        Stock.get_or_create(adresse=padding_address(tmp_adr), reference="DEVERMINAGE", defaults={
            "quantite": 10,
            "lot": "ABCDTEST",
            "contenant": "123456789"
        })

    def teardown_method(self, method):
        Magasin.delete().execute()
        Adresse.delete().execute()
        Stock.delete().execute()

    def test_emplacement_libre(self):
        emp_list = EmplacementView.libre()
        assert len(emp_list) == 48

    def test_emplacement_libre_produit_inconnu(self):
        emp_list = EmplacementView.libre("INCONNU")
        assert len(emp_list) == 48

    def test_emplacement_libre_produit_connu(self):
        emp_list = EmplacementView.libre("DEVERMINAGE")
        assert len(emp_list) == 48

    def test_emplacement_libre_produit_connu_st1(self):
        """Emplacement de ST1 sur produit DEVERMINAGE"""
        emp_list = EmplacementView.libre("DEVERMINAGE", "ST1")
        assert len(emp_list) == 14

    def test_emplacement_libre_produit_connu_st10(self):
        """Emplacement magasin inexistant sur produit DEVERMINAGE"""
        emp_list = EmplacementView.libre("DEVERMINAGE", "ST10")
        assert len(emp_list) == 0

    def test_emplacement_libre_magasin_st1(self):
        """Emplacement magasin ST1"""
        emp_list = EmplacementView.libre(magasin="ST1")
        assert len(emp_list) == 14

    def test_emplacement_libre_magasin_u01(self):
        """Emplacement magasin U01"""
        emp_list = EmplacementView.libre(magasin="U01")
        assert len(emp_list) == 28

    def test_emplacement_libre_magasin_dc1(self):
        """Emplacement magasin DC1"""
        emp_list = EmplacementView.libre(magasin="DC1")
        assert len(emp_list) == 6

    def test_emplacement_libre_magasin_st10(self):
        """Emplacement magasin ST10 inexistant"""
        emp_list = EmplacementView.libre(magasin="ST10")
        assert len(emp_list) == 0

    def test_emplacement_libre_format_boite_pass(self):
        """Emplacement libre avec boite pass"""
        emp_list = EmplacementView.libre(cformat="BOITE PASS")
        assert len(emp_list) == 20

    def test_emplacement_libre_format_boite_pass_st1(self):
        """Emplacement libre avec boite pass"""
        emp_list = EmplacementView.libre(magasin="ST1", cformat="BOITE PASS")
        assert len(emp_list) == 14

    def test_emplacement_libre_format_boite_pass_dc1(self):
        """Emplacement libre DC1 avec boite pass"""
        emp_list = EmplacementView.libre(magasin="DC1", cformat="BOITE PASS")
        assert len(emp_list) == 6

    def test_emplacement_libre_format_bac_1_7_bas(self):
        """Emplacement libre avec bac 1/7 bas"""
        emp_list = EmplacementView.libre(cformat="BAC 1/7 B")
        assert len(emp_list) == 14

    def test_emplacement_libre_format_bac_1_7_bas_u01(self):
        """Emplacement libre U01 avec bac 1/7 bas"""
        emp_list = EmplacementView.libre(magasin="U01", cformat="BAC 1/7 B")
        assert len(emp_list) == 14

    def test_emplacement_libre_format_bac_1_7_bas_st1(self):
        """Emplacement libre ST1 avec bac 1/7 bas"""
        emp_list = EmplacementView.libre(magasin="ST1", cformat="BAC 1/7 B")
        assert len(emp_list) == 0

    def test_emplacement_libre_produit_connu_st1_boite_pass(self):
        """Emplacement de ST1 sur produit DEVERMINAGE boite pass"""
        emp_list = EmplacementView.libre("DEVERMINAGE", "ST1", "BOITE PASS")
        assert len(emp_list) == 14

    def test_emplacement_libre_produit_connu_st1_adr_check_1(self):
        """Emplacement de ST1 sur produit DEVERMINAGE boite pass, on vérifie l'adresse"""
        emp_list = EmplacementView.libre("DEVERMINAGE", "ST1", "BOITE PASS")
        assert "ST1.  1.  3.  1.  1" in [x["adresse"] for x in emp_list]

    def test_emplacement_libre_produit_connu_st1_adr_check_2(self):
        """Emplacement de ST1 sur produit DEVERMINAGE boite pass, on vérifie l'adresse
        il y'a du produit à l'adresse  [ST1.  1.  5.  2.  1], doit être visible
        seulement en création de stock.
        """
        emp_list = EmplacementView.libre("DEVERMINAGE", "ST1", "BOITE PASS")
        assert "ST1.  1.  5.  2.  1" not in [x["adresse"] for x in emp_list]

    def test_emplacement_libre_produit_connu_st1_adr_creation(self):
        """Emplacement de ST1 sur produit DEVERMINAGE boite pass, on vérifie l'adresse
        il y'a du produit à l'adresse  [ST1.  1.  5.  2.  1], doit être visible
        seulement en création de stock.
        """
        emp_list = EmplacementView.libre("DEVERMINAGE", "ST1", "BOITE PASS", True)
        assert "ST1.  1.  3.  1.  1" in [x["adresse"] for x in emp_list]

    def test_emplacement_libre_produit_connu_u01_bac_1_7_bas_st1(self):
        """Emplacement de U01 sur produit DEVERMINAGE bas 1/7 bas"""
        emp_list = EmplacementView.libre("DEVERMINAGE", "U01", "BAC 1/7 B")
        assert len(emp_list) == 6

    def test_emplacement_libre_produit_connu_u01_bac_1_7_bas_st1(self):
        """Emplacement de DC1 sur produit DEVERMINAGE bac 1/7 bas"""
        emp_list = EmplacementView.libre("DEVERMINAGE", "DC1", "BAC 1/7 B")
        assert len(emp_list) == 0
