# flake8: noqa
import pytest
from peewee import DoesNotExist
from median.base import BaseModel
from median.migration import DbMigrator
from median.base import mysql_db
from median.models import Service, Patient, Sejour

from peewee import (
    AutoField,
    CharField,
    IntegerField,
    BooleanField,
    DateTimeField,
    SQL,
    ProgrammingError,
)


class TableEx(BaseModel):

    pk = AutoField(column_name="x_pk", primary_key=True)
    name = CharField(column_name="x_name", constraints=[SQL("DEFAULT ''")])
    quantity = IntegerField(column_name="x_quantity", constraints=[SQL("DEFAULT 0")])
    date_test = DateTimeField(
        column_name="x_date_test", constraints=[SQL("DEFAULT '0000-00-00 00:00:00'")]
    )
    is_ok = BooleanField(
        column_name="x_bool", null=True, constraints=[SQL("DEFAULT 0")]
    )

    class Meta:
        table_name = "test_table_ex"
        indexes = ((("name",), True),)


@pytest.fixture()
def rectest():
    print("Running method level setUp")
    rec_data = {}
    rec_data["mig"] = DbMigrator(mysql_db)

    yield rec_data

    print("Running method level tearDown")


class TestDbMigrator:
    def test_db_migrator(self, rectest):
        """Enable test migration"""
        rectest["mig"].migrate()

    def test_add_table(self, rectest):

        rectest["mig"].check_model("test_table_ex", TableEx)
        rectest["mig"].delete_model("test_table_ex", TableEx)

    def test_delete_missing_model(self, rectest):
        rectest["mig"].delete_model("test_table_ex", TableEx)

    def test_missing_field(self, rectest):
        res = rectest["mig"].check_field("test_table_ex", "x_bool")
        assert res == False

    def test_add_table_check_field(self, rectest):
        rectest["mig"].check_model("test_table_ex", TableEx)
        res = rectest["mig"].check_field("test_table_ex", "x_bool")
        assert res == True
        rectest["mig"].delete_model("test_table_ex", TableEx)

    def test_unknown_field_on_table(self, rectest):
        res = rectest["mig"].check_field("f_liste", "x_unknown")
        assert res == False

    def test_add_known_field_on_table(self, rectest):
        res = rectest["mig"].check_field(
            "f_liste",
            "x_known",
            "ALTER TABLE f_liste ADD COLUMN x_known VARCHAR(30) DEFAULT '';",
        )
        assert res == True
        res = rectest["mig"].delete_field("f_liste", "x_known")
        assert res == True

    def test_add_login_field_on_user(self, rectest):
        """Add login field on user"""
        res = rectest["mig"].check_field(
            "f_user",
            "x_import_flag",
            "ALTER TABLE f_user ADD COLUMN x_import_flag TINYINT(1) NULL DEFAULT 0;",
        )
        assert res == True

    def test_add_maintenance_field_on_user(self, rectest):
        """Add maintenance field on user"""
        res = rectest["mig"].check_field(
            "f_user",
            "x_maintenance",
            "ALTER TABLE f_user ADD COLUMN x_maintenance TINYINT(1) NULL DEFAULT 0;",
        )
        assert res == True

    def test_delete_deenova_wrd_migrate(self, rectest):
        """Delete Deenova ward create by migrate function"""
        srv = Service.get(code="DEENOVA")
        srv.delete_instance()

    def test_deenova_wrd(self, rectest):
        """Test if Deenova ward is defined on f_dest"""
        with pytest.raises(DoesNotExist):
            Service.get(code="DEENOVA")

    def test_create_service(self, rectest):
        """Test de creation d'uns service"""
        res = rectest["mig"].add_service("DEENOVA", "Unit test demo")
        assert res == True

    def test_delete_ipp_and_epidode(self, rectest):
        Patient.delete().where(Patient.ipp == "GLOBAL").execute()
        Sejour.delete().where(Sejour.ipp == "GLOBAL").execute()

    def test_add_global_ipp_episode(self, rectest):
        res = rectest["mig"].add_global_ipp_episode()
        assert res == True

    def test_delete_deenova_wrd(self, rectest):
        """Delete Deenova ward is defined on f_dest"""
        srv = Service.get(code="DEENOVA")
        srv.delete_instance()
