import asyncio
from pathlib import Path
from typing import List
import logging

PYTHON_VERSIONS = ["3.13"]
SCRIPT_DIR = Path(__file__).parent
REQ_DIR = SCRIPT_DIR


async def cleanup_old_files() -> None:
    """Remove existing requirement files before regenerating."""
    for file in REQ_DIR.glob("py*requirements.txt"):
        file.unlink()
        logging.info(f"Removed {file.name}")


async def compile_requirements(version: str, req_type: str = "prod") -> None:
    """Compile requirements for specific Python version and type."""
    base_cmd = "uv pip compile --strip-extras --generate-hashes"

    if req_type == "prod":
        req_in = REQ_DIR / "requirements.in"
        output = REQ_DIR / f"py{version}_requirements.txt"
        cmd = f"{base_cmd} {req_in} -o {output} --python-version {version}"
    else:
        req_in = REQ_DIR / "requirements.in"
        dev_in = REQ_DIR / "dev-requirements.in"
        output = REQ_DIR / f"py{version}_dev-requirements.txt"
        cmd = f"{base_cmd} {req_in} {dev_in} -o {output} --python-version {version}"

    logging.info(f"Compiling {req_type} requirements for Python {version}")
    process = await asyncio.create_subprocess_shell(cmd)
    await process.communicate()
    logging.info(f"Finished compiling {output.name}")


async def main():
    logging.basicConfig(level=logging.INFO)
    await cleanup_old_files()

    tasks: List[asyncio.Task] = []

    # Generate production requirements
    for version in PYTHON_VERSIONS:
        tasks.append(asyncio.create_task(compile_requirements(version, "prod")))
    await asyncio.gather(*tasks)

    # Generate development requirements
    tasks.clear()
    for version in PYTHON_VERSIONS:
        tasks.append(asyncio.create_task(compile_requirements(version, "dev")))
    await asyncio.gather(*tasks)


if __name__ == "__main__":
    asyncio.run(main())
