import time

import pytest
import os
from peewee import DoesNotExist

from median.base import BaseView, BaseViewException
from median.views import PrinterView, PrinterViewException
from median.models import Magasin, Printer, PrinterMag, PrinterType, PrinterLabel, User, PrinterCommand, PrinterCommandLabel

class TestViewPrinter:

    def setup_method(self, method):
        Magasin.delete().execute()
        Printer.delete().execute()

        self.user_1, _ = User.get_or_create(username="Testing User", defaults={
            "password": "123456789",
            "profil": "ECO-DEX"
        })

        self.mag_1, _ = Magasin.get_or_create(mag="ST1", type_mag="ACCED_ST1", defaults={
            "eco_type": "C",
            "type_machine": "ACCED_V2",
            "id_zone": 1000,
            "id_robot": 2000
        })

        self.mag_2, _ = Magasin.get_or_create(mag="ST2", type_mag="ACCED_ST2", defaults={
            "eco_type": "C",
            "type_machine": "ACCED_V2",
            "id_zone": 1000,
            "id_robot": 2000
        })

        self.mag_3, _ = Magasin.get_or_create(mag="ST3", type_mag="ACCED_ST3", defaults={
            "eco_type": "C",
            "type_machine": "ACCED_V2",
            "id_zone": 1000,
            "id_robot": 2000
        })

        self.prt_type_1, _ = PrinterType.get_or_create(name="Test 1", defaults={
            "lang": "ZPL",
            "density": 300,
        })

        self.prt_label_1, _ = PrinterLabel.get_or_create(name="Label Test 1", defaults={
            "width": 106,
            "height": 38
        })

        self.prt_label_2, _ = PrinterLabel.get_or_create(name="Label Test 2", defaults={
            "width": 86,
            "height": 25
        })

        self.printer_1, _ = Printer.get_or_create(name="TEST TCP 1 GX 430", defaults={
            "type_id": self.prt_type_1,
            "address": "tcp://127.0.0.1:9100",
            "current_label_id": self.prt_label_1,
            "option_cut": 1,
        })

        self.printer_mock, _ = Printer.get_or_create(name="TEST MOCK 1 GX 430", defaults={
            "type_id": self.prt_type_1,
            "address": "mock://127.0.0.1:9100",
            "current_label_id": self.prt_label_1,
            "option_cut": 1,
        })

        self.printer_mock2 = Printer.get_or_none(name="TEST MOCK 1 GX 430")

        self.printer_bad, _ = Printer.get_or_create(name="TEST BAD GX 430", defaults={
            "type_id": self.prt_type_1,
            "address": "usb://127.0.0.1:9100",
            "current_label_id": self.prt_label_1,
            "option_cut": 1,
        })

        self.prt_mag, _ = PrinterMag.get_or_create(printer_id=self.printer_1, mag_id=self.mag_1, defaults={
            "primary": 1
        })

        self.prt_mag2_1, _ = PrinterMag.get_or_create(printer_id=self.printer_1, mag_id=self.mag_2, defaults={
            "primary": 1
        })
        self.prt_mag2_2, _ = PrinterMag.get_or_create(printer_id=self.printer_mock, mag_id=self.mag_2, defaults={
            "primary": 0
        })
        self.prt_cmd_1, _ = PrinterCommand.get_or_create(code="DRUG_LABEL", defaults={
            "name": "Drugs labels"
        })
        self.cmd_lab_1, _ = PrinterCommandLabel.get_or_create(command_id=self.prt_cmd_1.pk, label_id=self.prt_label_1.pk, defaults={
            "print_code": "^XA%(name)s^XZ",
            "print_dict": "{'name': 'paracetamol'}"
        })

    def teardown_method(self, method):
        self.cmd_lab_1.delete_instance()
        self.prt_cmd_1.delete_instance()
        self.prt_mag.delete_instance()
        self.prt_mag2_1.delete_instance()
        self.prt_mag2_2.delete_instance()
        self.printer_1.delete_instance()
        self.printer_bad.delete_instance()
        self.printer_mock.delete_instance()
        self.prt_label_2.delete_instance()
        self.prt_label_1.delete_instance()
        self.prt_type_1.delete_instance()
        self.mag_3.delete_instance()
        self.mag_2.delete_instance()
        self.mag_1.delete_instance()
        self.user_1.delete_instance()
        Printer.delete().execute()
        Magasin.delete().execute()

    def test_init_printer_view(self,):
        prt = PrinterView()
        assert isinstance(prt, BaseView)

    def test_printers_list(self):
        """Retrieve all printers"""
        res = PrinterView.printers_list()
        assert len(res) == 3

    def test_printers_by_warehouse_without_warehouse(self):
        """Display printer list by warehouse"""
        prt = PrinterView()
        assert prt.printers_by_warehouse() == []

    def test_printers_by_warehouse_with_warehouse(self):
        """Display printer list by warehouse"""
        res = PrinterView.printers_by_warehouse(self.mag_1)
        assert len(res) == 1

    def test_printers_by_warehouse_with_warehouse_2(self):
        """Display printer list by warehouse"""
        res = PrinterView.printers_by_warehouse(self.mag_2)
        assert len(res) == 2

    def test_printers_by_warehouse_with_warehouse_3(self):
        """Display printer list by warehouse"""
        res = PrinterView.printers_by_warehouse(self.mag_3)
        assert len(res) == 0

    def test_printers_default_printer_without_warehouse(self):
        res = PrinterView.default_printer()
        assert res is None

    def test_printers_default_printer_with_warehouse(self):
        res = PrinterView.default_printer(self.mag_2)
        assert res == self.printer_1

    def test_printers_default_printer_with_warehouse_no(self):
        with pytest.raises(PrinterViewException) as e:
            res = PrinterView.default_printer(self.mag_3)
        assert 'no default printer for' in str(e)

    def test_printers_default_printer_without_label(self):
        res = PrinterView.printers_by_label()
        assert res == []

    def test_printers_default_printer_with_label_1(self):
        res = PrinterView.printers_by_label(self.printer_1.current_label_id)
        assert len(res) == 3
        assert res[0] == self.printer_bad
        assert res[1] == self.printer_mock
        assert res[2] == self.printer_1

    def test_printers_default_printer_with_label_2(self):
        res = PrinterView.printers_by_label(self.prt_label_2)
        assert len(res) == 0

    def test_printers_search_by_label_0(self):
        res = PrinterView.printers_by_command()
        assert len(res) == 0

    def test_printers_search_by_label_1(self):
        res = PrinterView.printers_by_command(self.prt_cmd_1)
        assert len(res) == 3
        assert res[0] == self.printer_bad
        assert res[0].current_label_id == self.printer_1.current_label_id
        assert res[1] == self.printer_mock
        assert res[1].current_label_id == self.printer_1.current_label_id
        assert res[2] == self.printer_1
        assert res[2].current_label_id == self.printer_1.current_label_id

    def test_printer_context_missing_printername(self):
        with pytest.raises(PrinterViewException) as e:
            with PrinterView() as p:
                pass
        assert "Printer name is required" in str(e)

    def test_printer_context_printer_bad(self):
        with pytest.raises(PrinterViewException) as e:
            with PrinterView(self.printer_bad) as p:
                pass
        assert "Printer must be in TCP" in str(e)

    def test_printer_context_printer_retry(self):
        with pytest.raises(PrinterViewException) as e:
            with PrinterView(self.printer_mock, 0, self.user_1) as p:
                p.send("blah blah")

        assert "Number of retry exceeded" in str(e)

    def test_printer_context_printer_mock(self):

        with PrinterView(self.printer_mock, 3, self.user_1) as p:
            assert p.send("blah blah") == 9
            assert p.send(None) is None
            assert p.send("\n") == 1
            assert p.send(b"\x65\x78\x62\x32\x65") == 5
            assert p.send("\n") == 1
            assert p.send(123456789) == 9

    def test_printer_context_printer_mock_get(self):
        self.printer_mock2.current_label_id = int(self.prt_label_1.pk)
        with PrinterView(self.printer_mock2, 3, self.user_1) as p:
            assert p.send("blah blah") == 9
            assert p.send(None) is None
            assert p.send("\n") == 1
            assert p.send(b"\x65\x78\x62\x32\x65") == 5
            assert p.send("\n") == 1
            assert p.send(123456789) == 9

    @pytest.mark.xfail("'CI' in os.environ")
    def test_printer_context_printer_10(self):

        with PrinterView(self.printer_1, 3, self.user_1) as p:
            assert p.send("blah blah", "Encore du blah blah") == 9
            assert p.send(None, "Oups") is None
            assert p.send("\n") == 1
            assert p.send(b"\x65\x78\x62\x32\x65", "test en hexadecimal") == 5
            assert p.send("\n") == 1
            assert p.send(123456789, "On essai un entier") == 9

    def test_printer_retrieve_template_not_exists(self):
        with pytest.raises(PrinterViewException) as e:
            with PrinterView(self.printer_mock, 3, self.user_1) as p:
                tmpl, default = p.printer_template("UNKNOWN")
        assert "command name UNKNOWN not exists" in str(e)

    def test_printer_retrieve_template_exists(self):
        with PrinterView(self.printer_mock, 3, self.user_1) as p:
            tmpl, default = p.printer_template(self.prt_cmd_1.code)
            assert tmpl == self.cmd_lab_1.print_code
            assert default == self.cmd_lab_1.print_dict

    def test_printer_retrieve_template_exists_bad_label(self):
        self.printer_mock.current_label_id = self.prt_label_2
        with pytest.raises(PrinterViewException) as e:
            with PrinterView(self.printer_mock, 3, self.user_1) as p:
                tmpl, default = p.printer_template(self.prt_cmd_1.code)
        assert "No command DRUG_LABEL label found" in str(e)
