# flake8: noqa
import pytest
from peewee import DoesNotExist, IntegrityError
from median.models import Product, Service, ReferencePerService
from median.views import Seuil as vSeuil
from median.base import BaseViewException


@pytest.fixture()
def rectest():
    print("Running method level setUp")
    rec_data = {}
    Product.delete().where(Product.reference=="REAP0001").execute()
    Service.delete().where(Service.code=="0001").execute()

    p1 = Product(reference="REAP0001", designation="Produit reappro 1")
    p1.save()
    rec_data["p1"] = p1

    w1 = Service(code="0001", libelle="Service 1", type_dest="SERVICE")
    w1.save()
    rec_data["w1"] = w1

    yield rec_data
    print("Running method level tearDown")
    w1.delete_instance()
    p1.delete_instance()

def test_link_product_ward(rectest):

    rps = ReferencePerService()
    rps.ref_pk = rectest['p1'].pk
    rps.dest_pk = rectest['w1'].pk
    rps.save()
    assert rps.pk > 0
    rps.delete_instance()

def test_link_product_ward_wrong_ref(rectest):

    with pytest.raises(IntegrityError) as e:
        rps = ReferencePerService()
        rps.ref_pk = 0
        rps.dest_pk = rectest['w1'].pk
        rps.save()
    assert "a foreign key constraint fails" in str(e)

def test_link_product_ward_wrong_ward(rectest):

    with pytest.raises(IntegrityError) as e:
        rps = ReferencePerService()
        rps.ref_pk = rectest['p1'].pk
        rps.dest_pk = 0
        rps.save()
    assert "a foreign key constraint fails" in str(e)


def test_link_product_ward_backref(rectest):

    rps = ReferencePerService()
    rps.ref_pk = rectest['p1'].pk
    rps.dest_pk = rectest['w1'].pk
    rps.save()
    assert rps.pk > 0

    pro = ReferencePerService.select(
            ReferencePerService, Product, Service
        ).join(
            Product
        ).switch(ReferencePerService).join(
            Service
        ).where(Product.pk == rectest['p1'].pk).get()

    assert pro.ref_pk.pk == rectest['p1'].pk
    assert pro.dest_pk.pk == rectest['w1'].pk
    assert pro.distribution_type == "N"
    assert pro.eco_type == "E"

    rps.delete_instance()


def test_link_product_ward_global(rectest):

    rps = ReferencePerService()
    rps.ref_pk = rectest['p1'].pk
    rps.dest_pk = rectest['w1'].pk
    rps.distribution_type = "G"
    rps.save()
    assert rps.pk > 0

    pro = ReferencePerService.select(
            ReferencePerService, Product, Service
        ).join(
            Product
        ).switch(ReferencePerService).join(
            Service
        ).where(Product.pk == rectest['p1'].pk).get()

    assert pro.ref_pk.pk == rectest['p1'].pk
    assert pro.dest_pk.pk == rectest['w1'].pk
    assert pro.distribution_type == "G"
    assert pro.eco_type == "E"

    rps.delete_instance()


def test_link_product_ward_eco_type_pick(rectest):

    rps = ReferencePerService()
    rps.ref_pk = rectest['p1'].pk
    rps.dest_pk = rectest['w1'].pk
    rps.distribution_type = "G"
    rps.eco_type = "C"
    rps.save()
    assert rps.pk > 0

    pro = ReferencePerService.select(
            ReferencePerService, Product, Service
        ).join(
            Product
        ).switch(ReferencePerService).join(
            Service
        ).where(Product.pk == rectest['p1'].pk).get()

    assert pro.ref_pk.pk == rectest['p1'].pk
    assert pro.dest_pk.pk == rectest['w1'].pk
    assert pro.distribution_type == "G"
    assert pro.eco_type == "C"

    rps.delete_instance()


def test_link_product_ward_eco_type_cut(rectest):

    rps = ReferencePerService()
    rps.ref_pk = rectest['p1'].pk
    rps.dest_pk = rectest['w1'].pk
    rps.distribution_type = "G"
    rps.eco_type = "T"
    rps.save()
    assert rps.pk > 0

    pro = ReferencePerService.select(
            ReferencePerService, Product, Service
        ).join(
            Product
        ).switch(ReferencePerService).join(
            Service
        ).where(Product.pk == rectest['p1'].pk).get()

    assert pro.ref_pk.pk == rectest['p1'].pk
    assert pro.dest_pk.pk == rectest['w1'].pk
    assert pro.distribution_type == "G"
    assert pro.eco_type == "T"

    rps.delete_instance()


def test_link_product_ward_bad_distribution_type(rectest):

    with pytest.raises(ValueError) as e:
        rps = ReferencePerService()
        rps.ref_pk = rectest['p1'].pk
        rps.dest_pk = rectest['w1'].pk
        rps.distribution_type = "Z"
        rps.save()

    assert "bad value Z" in str(e)

def test_link_product_ward_bad_eco_type(rectest):

    with pytest.raises(ValueError) as e:
        rps = ReferencePerService()
        rps.ref_pk = rectest['p1'].pk
        rps.dest_pk = rectest['w1'].pk
        rps.eco_type = "Z"
        rps.save()

    assert "bad value Z" in str(e)
