import time

import pytest
from peewee import DoesNotExist

from median.base import BaseView, BaseViewException
from median.constant import EtatListe, PatientGlobal, TypeListe, TypeServiListe, EquipmentType
# from median.views import Dispensation, Items, Dispensation_Suggestion
from median.models import Cip, Endpoint, Magasin, Poste, Product, Service, Ucd
from median.models import Stock, ListeModel, ListeItemModel, WorkItemModel, Seuil, Compteur
from median.views import  ExternalView, ExternalViewException
from datetime import datetime

# from datetime import datetime, date


@pytest.fixture()
def rectest():
    print("Running method level setUp")
    Product.delete().execute()
    WorkItemModel.delete().execute()
    ListeItemModel.delete().execute()
    ListeModel.delete().execute()
    rec_data = {}
    stk_data = []

    Compteur.get_or_create(cle="EXTERNAL_INVENTORY", defaults={
        "val": 1
    })

    pos99, _ = Poste.get_or_create(poste="EXT_99", defaults={
        "pc": "SERVEUR"
    })

    pos00, _ = Poste.get_or_create(poste="EXT_00", defaults={
        "pc": "SERVEUR"
    })

    whs99, _ = Magasin.get_or_create(mag="E99", defaults={
        "type_mag": pos99.poste,
        "eco_type": "L",
        "id_robot": "R01",
        "id_zone": "TMP",
    })

    whs00, _ = Magasin.get_or_create(mag="E00", defaults={
        "type_mag": pos00.poste,
        "eco_type": "L",
        "id_robot": "R00",
        "id_zone": "TMP",
    })

    ward_00, _ = Service.get_or_create(code="WARD00", defaults={
        "libelle": "Test Ward",
    })

    pr1 = Product()
    pr1.reference = "ECO6547"
    pr1.designation = "Eco Medoc"
    pr1.save()

    pr2 = Product()
    pr2.reference = "ECO9876"
    pr2.designation = "Eco Medoc 2"
    pr2.save()

    pr3 = Product()
    pr3.reference = "ECO8754"
    pr3.designation = "Eco Medoc 3"
    pr3.save()

    pr4 = Product()
    pr4.reference = "ECO6521"
    pr4.designation = "Eco Medoc 4"
    pr4.save()

    seuil1 = Seuil()
    seuil1.reference = pr1.reference
    seuil1.zone = pos00.poste
    seuil1.fraction = 100
    seuil1.save()

    ucd1 = Ucd()
    ucd1.ucd = "3400800000001"
    ucd1.reference = pr1.reference
    ucd1.save()

    cip1 = Cip()
    cip1.ucd = ucd1.ucd
    cip1.cip = "3400900000002"
    cip1.save()

    cip2 = Cip()
    cip2.ucd = ucd1.ucd
    cip2.cip = "3400900000003"
    cip2.save()

    for n in range(8):
        stk = Stock()
        stk.reference = pr1.reference
        stk.adresse = "%s." % whs00.mag
        stk.magasin = whs00.mag
        stk.lot = "ABCDS"
        stk.quantite = 25
        stk.date_entree = datetime.now()
        stk.save()
        stk_data.append(stk)

    rec_data["ext99"] = pos99
    rec_data["ext00"] = pos00
    rec_data["whs99"] = whs99
    rec_data["whs00"] = whs00
    rec_data["ward_00"] = ward_00
    rec_data["pr1"] = pr1
    rec_data["pr2"] = pr2
    rec_data["pr3"] = pr3
    rec_data["pr4"] = pr4
    rec_data["ucd1"] = ucd1
    rec_data["cip1"] = cip1

    yield rec_data

    print("Running method level tearDown")
    for x in stk_data:
        x.delete_instance()
    cip1.delete_instance()
    cip2.delete_instance()
    ucd1.delete_instance()
    seuil1.delete_instance()
    pr4.delete_instance()
    pr3.delete_instance()
    pr2.delete_instance()
    pr1.delete_instance()
    ward_00.delete_instance()
    whs99.delete_instance()
    whs00.delete_instance()
    pos99.delete_instance()
    pos00.delete_instance()

class TestViewExternal:
    def test_init_externalview_without_poste(self, rectest):
        """
        Test d'initialisation de la classe
        """
        with pytest.raises(BaseViewException) as e:
            ext = ExternalView()
        assert "poste must be required" in str(e.value)

    def test_init_externalview_unknown_poste(self, rectest):
        """
        Test d'initialisation de la classe
        """
        with pytest.raises(BaseViewException) as e:
            ext = ExternalView("UNKNOWN")
        assert "does not exists" in str(e.value)

    def test_init_externalview(self, rectest):
        """Test instance with known Riedl"""
        ext = ExternalView(rectest["ext99"].poste)
        assert ext.poste.poste == rectest["ext99"].poste

    def test_externalview_warehouse_99(self, rectest):
        """Test instance with known Riedl"""
        ext = ExternalView(rectest["ext99"].poste)
        assert ext.poste.poste == rectest["ext99"].poste
        assert len(ext.warehouses) == 1

    def test_riedlview_generate_inventory_l00(self, rectest):
        ext = ExternalView(rectest["ext00"].poste)
        list_id, nb_ref = ext.generate_inventory()

        assert list_id.pk > 0
        assert nb_ref == 1
        assert list_id.etat == "V"
        assert list_id.mode == "I"
        assert list_id.zone_deb == EquipmentType.EXTERNE.value
        assert list_id.zone_fin == rectest["whs00"].type_mag

        list_id.delete_instance()

    def test_riedlview_generate_inventory_l99(self, rectest):
        ext = ExternalView(rectest["ext99"].poste)
        list_id, nb_ref = ext.generate_inventory()

        assert list_id.pk > 0
        assert nb_ref == 0

        list_id.delete_instance()
