import time

import pytest
from peewee import DoesNotExist

from median.base import BaseView, BaseViewException
from median.constant import EtatListe, PatientGlobal, TypeListe, TypeServiListe, EcoType
# from median.views import Dispensation, Items, Dispensation_Suggestion
from median.models import Cip, Adresse, Magasin, Poste, Product, Service, Ucd
from median.models import Stock, ListeModel, ListeItemModel, WorkItemModel, Seuil, Compteur
from median.views import GeneratorView
from datetime import datetime

# from datetime import datetime, date

@pytest.fixture()
def rectest():
    print("Running method level setUp")
    Poste.delete().where(Poste.poste << [
        "ASTUS_XL", "ASTUS_6", "ASTUS_7", "ASTUS_8",
        "ASTUS_9", "ASTUS_10", "ASTUS_11"]).execute()
    Magasin.delete().where(Magasin.eco_type=="A").execute()
    Adresse.delete().where(Adresse.magasin << [
        "UXL", "U06", "U07", "U08",
        "U09", "U10", "U11"]).execute()

    Compteur.get_or_create(cle="CONTENANT_PASS", defaults={
        "val": 1
    })

    rec_data = {}
    pos99, _ = Poste.get_or_create(poste="ASTUS_99", defaults={
        "pc": "SERVEUR"
    })
    whs99, _ = Magasin.get_or_create(mag="A99", defaults={
        "type_mag": pos99.poste,
        "eco_type": "A",
        "id_robot": "A01",
        "id_zone": "T01",
    })

    rec_data["pos99"] = pos99
    rec_data["whs99"] = whs99

    yield rec_data

    print("Running method level tearDown")
    whs99.delete_instance()
    pos99.delete_instance()


class TestViewGenerator:
    def test_init_generator_failed(self):
        with pytest.raises(BaseViewException) as e:
            ast = GeneratorView()
        assert "Warehouse code is required" in str(e.value)

    def test_init_generator_astu99(self, rectest):
        ast = GeneratorView(rectest["whs99"].mag)
        assert ast.warehouse.pk > 0
        assert ast.warehouse.mag == rectest["whs99"].mag
        assert ast.warehouse.eco_type == rectest["whs99"].eco_type

    def test_create_exists_astus(self, rectest):
        ast = GeneratorView(rectest["whs99"].mag)
        assert ast.warehouse.pk > 0
        with pytest.raises(BaseViewException) as e:
            ast.astus_add_xl("Astus 99", "")

    def test_create_new_astus_xl(self, rectest):
        ast = GeneratorView("UXL", EcoType.Astus.value)
        ast.astus_add_xl("ASTUS_XL", "Astus XL", "AXL", "TXL")
        pos = Poste.get(poste="ASTUS_XL")
        mag = Magasin.get(mag="UXL")

        assert pos.proc_pid == 0
        assert mag.type_mag == pos.poste
        assert mag.dim_2 == 14
        assert mag.dim_3 == 6
        assert mag.dim_4 == 1
        assert mag.dim_5 == 0
        assert mag.id_robot == "AXL"
        assert mag.id_zone == "TXL"
        assert mag.libelle == "Astus XL"

        addrs = Adresse.select(Adresse).where(Adresse.magasin==mag.mag)
        assert len(addrs) == 77

        for adr in addrs:
            assert adr.etat == "L"
            assert adr.contenant != ""
            assert adr.format.startswith("PIL")

    def test_create_new_astus_V6(self, rectest):
        ast = GeneratorView("U06", EcoType.Astus.value)
        ast.astus_add_v6("ASTUS_6", "Astus 6", "A06", "T06")
        pos = Poste.get(poste="ASTUS_6")
        mag = Magasin.get(mag="U06")

        assert pos.proc_pid == 0
        assert mag.type_mag == pos.poste
        assert mag.dim_2 == 23
        assert mag.dim_3 == 6
        assert mag.dim_4 == 2
        assert mag.dim_5 == 0
        assert mag.id_robot == "A06"
        assert mag.id_zone == "T06"
        assert mag.libelle == "Astus 6"

        addrs = Adresse.select(Adresse).where(Adresse.magasin==mag.mag)
        assert len(addrs) == 254

        for adr in addrs:
            assert adr.etat == "L"
            assert adr.contenant != ""
            assert adr.format.startswith("BAC")

    def test_create_new_astus_V7(self, rectest):
        ast = GeneratorView("U07", EcoType.Astus.value)
        ast.astus_add_v7("ASTUS_7", "Astus 7", "A07", "T07")
        pos = Poste.get(poste="ASTUS_7")
        mag = Magasin.get(mag="U07")

        assert pos.proc_pid == 0
        assert mag.type_mag == pos.poste
        assert mag.dim_2 == 23
        assert mag.dim_3 == 7
        assert mag.dim_4 == 2
        assert mag.dim_5 == 0
        assert mag.id_robot == "A07"
        assert mag.id_zone == "T07"
        assert mag.libelle == "Astus 7"

        addrs = Adresse.select(Adresse).where(Adresse.magasin==mag.mag)
        assert len(addrs) == 300

        for adr in addrs:
            assert adr.etat == "L"
            assert adr.contenant != ""
            assert adr.format.startswith("BAC")

    def test_create_new_astus_V8(self, rectest):
        ast = GeneratorView("U08", EcoType.Astus.value)
        ast.astus_add_v8("ASTUS_8", "Astus 8", "A08", "T08")
        pos = Poste.get(poste="ASTUS_8")
        mag = Magasin.get(mag="U08")

        assert pos.proc_pid == 0
        assert mag.type_mag == pos.poste
        assert mag.dim_2 == 23
        assert mag.dim_3 == 8
        assert mag.dim_4 == 2
        assert mag.dim_5 == 0
        assert mag.id_robot == "A08"
        assert mag.id_zone == "T08"
        assert mag.libelle == "Astus 8"

        addrs = Adresse.select(Adresse).where(Adresse.magasin==mag.mag)
        assert len(addrs) == 346

        for adr in addrs:
            assert adr.etat == "L"
            assert adr.contenant != ""
            assert adr.format.startswith("BAC")

    def test_create_new_astus_V9(self, rectest):
        ast = GeneratorView("U09", EcoType.Astus.value)
        ast.astus_add_v9("ASTUS_9", "Astus 9", "A09", "T09")
        pos = Poste.get(poste="ASTUS_9")
        mag = Magasin.get(mag="U09")

        assert pos.proc_pid == 0
        assert mag.type_mag == pos.poste
        assert mag.dim_2 == 23
        assert mag.dim_3 == 9
        assert mag.dim_4 == 2
        assert mag.dim_5 == 0
        assert mag.id_robot == "A09"
        assert mag.id_zone == "T09"
        assert mag.libelle == "Astus 9"

        addrs = Adresse.select(Adresse).where(Adresse.magasin==mag.mag)
        assert len(addrs) == 392

        for adr in addrs:
            assert adr.etat == "L"
            assert adr.contenant != ""
            assert adr.format.startswith("BAC")

    def test_create_new_astus_V10(self, rectest):
        ast = GeneratorView("U10", EcoType.Astus.value)
        ast.astus_add_v10("ASTUS_10", "Astus 10", "A10", "T10")
        pos = Poste.get(poste="ASTUS_10")
        mag = Magasin.get(mag="U10")

        assert pos.proc_pid == 0
        assert mag.type_mag == pos.poste
        assert mag.dim_2 == 23
        assert mag.dim_3 == 10
        assert mag.dim_4 == 2
        assert mag.dim_5 == 0
        assert mag.id_robot == "A10"
        assert mag.id_zone == "T10"
        assert mag.libelle == "Astus 10"

        addrs = Adresse.select(Adresse).where(Adresse.magasin==mag.mag)
        assert len(addrs) == 438

        for adr in addrs:
            assert adr.etat == "L"
            assert adr.contenant != ""
            assert adr.format.startswith("BAC")

    def test_create_new_astus_V11(self, rectest):
        ast = GeneratorView("U11", EcoType.Astus.value)
        ast.astus_add_v11("ASTUS_11", "Astus 11", "A11", "T11")
        pos = Poste.get(poste="ASTUS_11")
        mag = Magasin.get(mag="U11")

        assert pos.proc_pid == 0
        assert mag.type_mag == pos.poste
        assert mag.dim_2 == 23
        assert mag.dim_3 == 11
        assert mag.dim_4 == 2
        assert mag.dim_5 == 0
        assert mag.id_robot == "A11"
        assert mag.id_zone == "T11"
        assert mag.libelle == "Astus 11"

        addrs = Adresse.select(Adresse).where(Adresse.magasin==mag.mag)
        assert len(addrs) == 484

        for adr in addrs:
            assert adr.etat == "L"
            assert adr.contenant != ""
            assert adr.format.startswith("BAC")
