# flake8: noqa
import pytest
from peewee import DoesNotExist
from median.base import BaseViewException, BaseViewDuplicate
from median.views import Emplacement, Adresse


class TestObjectAdresse:

    def test_init_adresse_sans_magasin(self):
        """Initialise une adresse sans magasin, erreur de création"""
        with pytest.raises(BaseViewException):
            emp = Adresse('ST1.  1.  1.  1.  1')
            emp.creer()

    def test_init_adresse_avec_magasin(self):
        """Creation de l'adresse"""
        emp = Adresse('ST1.  1.  1.  1.  1', 'ST1')
        emp.creer()
        assert emp.id > 0

    def test_init_adresse_en_doublons(self):
        """L'emplacement est en doublons, nous ne pouvons pas le réecréer"""
        with pytest.raises(BaseViewException):
            emp2 = Adresse('ST1.  1.  1.  1.  1', 'ST1')
            emp2.creer()

    def test_init_emplacement_fail(self):
        """Test création emplacement qui doit retourner une erreur"""
        with pytest.raises(BaseViewException):
            emp = Emplacement()
            emp.creer()

    def test_supprime_adresse(self):
        """Suppression de l'emplacement créé pour les tests"""
        emp = Adresse('ST1.  1.  1.  1.  1')
        assert emp.supprimer()

    def test_supprime_adresse_inexistante(self):
        """Suppression de l'emplacement qui n'existe pas"""
        with pytest.raises(BaseViewException):
            emp = Adresse('ST9.  0.  0.  0.  0')
            emp.supprimer()


class TestObjectEmplacement:

    def test_emplacement_libre(self):
        emp_list = Emplacement.libre()
        assert len(emp_list) == 14

    def test_emplacement_libre_produit_inconnu(self):
        emp_list = Emplacement.libre('INCONNU')
        assert len(emp_list) == 14

    def test_emplacement_libre_produit_connu(self):
        emp_list = Emplacement.libre('DEVERMINAGE')
        assert len(emp_list) == 15

    def test_emplacement_libre_produit_connu_st1(self):
        """Emplacement de ST1 sur produit DEVERMINAGE"""
        emp_list = Emplacement.libre('DEVERMINAGE', 'ST1')
        assert len(emp_list) == 2

    def test_emplacement_libre_produit_connu_st10(self):
        """Emplacement magasin inexistant sur produit DEVERMINAGE"""
        emp_list = Emplacement.libre('DEVERMINAGE', 'ST10')
        assert len(emp_list) == 0

    def test_emplacement_libre_magasin_st1(self):
        """Emplacement magasin ST1"""
        emp_list = Emplacement.libre(magasin='ST1')
        assert len(emp_list) == 2

    def test_emplacement_libre_magasin_u01(self):
        """Emplacement magasin U01"""
        emp_list = Emplacement.libre(magasin='U01')
        assert len(emp_list) == 7

    def test_emplacement_libre_magasin_dc1(self):
        """Emplacement magasin DC1"""
        emp_list = Emplacement.libre(magasin='DC1')
        assert len(emp_list) == 6

    def test_emplacement_libre_magasin_st10(self):
        """Emplacement magasin ST10 inexistant"""
        emp_list = Emplacement.libre(magasin='ST10')
        assert len(emp_list) == 0

    def test_emplacement_libre_format_boite_pass(self):
        """Emplacement libre avec boite pass"""
        emp_list = Emplacement.libre(cformat='BOITE PASS')
        assert len(emp_list) == 8

    def test_emplacement_libre_format_boite_pass_st1(self):
        """Emplacement libre avec boite pass"""
        emp_list = Emplacement.libre(magasin='ST1', cformat='BOITE PASS')
        assert len(emp_list) == 2

    def test_emplacement_libre_format_boite_pass_dc1(self):
        """Emplacement libre DC1 avec boite pass"""
        emp_list = Emplacement.libre(magasin='DC1', cformat='BOITE PASS')
        assert len(emp_list) == 6

    def test_emplacement_libre_format_bac_1_7_bas(self):
        """Emplacement libre avec bac 1/7 bas"""
        emp_list = Emplacement.libre(cformat='BAC 1/7 B')
        assert len(emp_list) == 6

    def test_emplacement_libre_format_bac_1_7_bas_u01(self):
        """Emplacement libre U01 avec bac 1/7 bas"""
        emp_list = Emplacement.libre(magasin='U01', cformat='BAC 1/7 B')
        assert len(emp_list) == 6

    def test_emplacement_libre_format_bac_1_7_bas_st1(self):
        """Emplacement libre ST1 avec bac 1/7 bas"""
        emp_list = Emplacement.libre(magasin='ST1', cformat='BAC 1/7 B')
        assert len(emp_list) == 0

    def test_emplacement_libre_produit_connu_st1_boite_pass(self):
        """Emplacement de ST1 sur produit DEVERMINAGE boite pass"""
        emp_list = Emplacement.libre('DEVERMINAGE', 'ST1', 'BOITE PASS')
        assert len(emp_list) == 2

    def test_emplacement_libre_produit_connu_st1_adr_check_1(self):
        """Emplacement de ST1 sur produit DEVERMINAGE boite pass, on vérifie l'adresse"""
        emp_list = Emplacement.libre('DEVERMINAGE', 'ST1', 'BOITE PASS')
        assert 'ST1.  1.  5.  2.  3' in [x['adresse'] for x in emp_list]

    def test_emplacement_libre_produit_connu_st1_adr_check_2(self):
        """Emplacement de ST1 sur produit DEVERMINAGE boite pass, on vérifie l'adresse
           il y'a du produit à l'adresse  [ST1.  1.  5.  2.  1], doit être visible
           seulement en création de stock.
        """
        emp_list = Emplacement.libre('DEVERMINAGE', 'ST1', 'BOITE PASS')
        assert 'ST1.  1.  5.  2.  1' not in [x['adresse'] for x in emp_list]

    def test_emplacement_libre_produit_connu_st1_adr_creation(self):
        """Emplacement de ST1 sur produit DEVERMINAGE boite pass, on vérifie l'adresse
           il y'a du produit à l'adresse  [ST1.  1.  5.  2.  1], doit être visible
           seulement en création de stock.
        """
        emp_list = Emplacement.libre('DEVERMINAGE', 'ST1', 'BOITE PASS', True)
        assert 'ST1.  1.  5.  2.  1' in [x['adresse'] for x in emp_list]

    def test_emplacement_libre_produit_connu_u01_bac_1_7_bas_st1(self):
        """Emplacement de U01 sur produit DEVERMINAGE bas 1/7 bas"""
        emp_list = Emplacement.libre('DEVERMINAGE', 'U01', 'BAC 1/7 B')
        assert len(emp_list) == 6

    def test_emplacement_libre_produit_connu_u01_bac_1_7_bas_st1(self):
        """Emplacement de DC1 sur produit DEVERMINAGE bac 1/7 bas"""
        emp_list = Emplacement.libre('DEVERMINAGE', 'DC1', 'BAC 1/7 B')
        assert len(emp_list) == 0
