from peewee import DoesNotExist
from .utils import logger
from .models import (
    PrescriptionItem, MagasinService, LogAstus, TopFive,
    HistoriqueMultidoses, MessageAstus, FListeError,
    Service, WebviewTaquin, WebviewDemiTaquin,
    WebviewEmplacement, WebviewPatientHisto
)


class DbMigrator:

    def __init__(self, db_instance):
        """Initilise the migraotr class with the database selected"""
        self.db = db_instance
        logger.debug('Initilize migrator instance')

    def migrate(self):
        """Launch the migration script"""
        self.check_model('f_prescription_item', PrescriptionItem)
        self.check_model('f_mag_dest', MagasinService)
        self.check_model('f_log_astus', LogAstus)
        self.check_model('f_top5', TopFive)
        self.check_model('f_histo_multidoses', HistoriqueMultidoses)
        self.check_model('f_message_astus', MessageAstus)
        self.check_model('f_liste_error', FListeError)
        self.check_field('f_user', 'x_astup', "ALTER TABLE f_user ADD COLUMN x_astup tinyint(1) DEFAULT '0';")
        self.check_field('f_user', 'x_inventaire', "ALTER TABLE f_user ADD COLUMN x_inventaire tinyint(1) DEFAULT '0';")
        self.check_field('f_user', 'x_login', "ALTER TABLE f_user ADD COLUMN x_login VARCHAR(30) DEFAULT '';")
        self.check_field('f_user', 'x_badge', "ALTER TABLE f_user ADD COLUMN x_badge varchar(16) DEFAULT '';")
        self.check_field('f_user', 'x_email', "ALTER TABLE f_user ADD COLUMN x_email VARCHAR(320) DEFAULT '';")
        self.check_field('f_user', 'x_import_flag',
                         "ALTER TABLE f_user ADD COLUMN x_import_flag TINYINT(1) NULL DEFAULT 0;")
        self.check_field('f_user', 'x_maintenance',
                         "ALTER TABLE f_user ADD COLUMN x_maintenance TINYINT(1) NULL DEFAULT 0;")
        self.check_field('f_profil', 'x_visu', "ALTER TABLE f_profil ADD COLUMN x_visu TINYINT(1) NULL DEFAULT 0;")
        self.check_field('f_profil', 'x_edit', "ALTER TABLE f_profil ADD COLUMN x_edit TINYINT(1) NULL DEFAULT 0;")
        self.check_field('f_ref', 'x_com_med', "ALTER TABLE f_ref ADD COLUMN x_com_med TEXT DEFAULT '';")
        self.check_field('f_ref', 'x_bac', "ALTER TABLE f_ref ADD COLUMN x_bac INT(11) NULL DEFAULT 0;")
        # Table f_dest for Carnet and Riedl Exit
        self.check_field('f_dest', 'x_type_carnet',
                         "ALTER TABLE f_dest ADD COLUMN x_type_carnet INT(11) NULL DEFAULT '1'")
        self.check_field('f_dest', 'x_secteur', "ALTER TABLE f_dest ADD COLUMN x_secteur VARCHAR(40) NULL DEFAULT ''")
        self.check_field('f_dest', 'x_adr1', "ALTER TABLE f_dest ADD COLUMN x_adr1 VARCHAR(40) NULL DEFAULT ''")
        self.check_field('f_dest', 'x_adr2', "ALTER TABLE f_dest ADD COLUMN x_adr2 VARCHAR(40) NULL DEFAULT ''")
        self.check_field('f_dest', 'x_adr3', "ALTER TABLE f_dest ADD COLUMN x_adr3 VARCHAR(40) NULL DEFAULT ''")
        self.check_field('f_dest', 'x_adr4', "ALTER TABLE f_dest ADD COLUMN x_adr4 VARCHAR(40) NULL DEFAULT ''")
        self.check_field('f_mag', 'x_split_liste',
                         "ALTER TABLE f_mag ADD COLUMN x_split_liste TINYINT(1) NULL DEFAULT 0;")
        self.check_field('f_mag', 'x_libelle',
                         "ALTER TABLE f_mag ADD COLUMN x_libelle varchar(35) NOT NULL DEFAULT '';")
        self.check_field('f_poste', 'x_proc_pid', "ALTER TABLE f_poste ADD COLUMN x_proc_pid INT(11) NULL DEFAULT 0;")

        self.add_service('DEENOVA', 'Service de test')
        # TODO: Drop all view before restore (force mode ?)
        logger.info('Check views')
        self.check_sql_view('webview_taquin', WebviewTaquin)
        self.check_sql_view('webview_demi_taquin', WebviewDemiTaquin)
        self.check_sql_view('webview_emplacement', WebviewEmplacement)
        self.check_sql_view('webview_patient_histo', WebviewPatientHisto)
        logger.info('End check views')

    def check_model(self, table_name, model_class):
        """Check if table exists, and create it if missing"""
        if not self.db.table_exists(table_name):
            logger.warning('Table %s does not exists, we create it from %s' % (table_name, model_class))
            self.db.create_tables([model_class])
        else:
            logger.info('Table %s exists' % (table_name,))

        return True

    def check_sql_view(self, view_name, model_class):
        """Check if view exists, if not create it"""
        vws = [v.name for v in self.db.get_views()]
        if view_name not in vws:
            logger.info(' -> View %s is missing, we create it' % view_name)
            kls = model_class()
            kls.create_view()

    def check_field(self, table_name, field_name, ddl_str=None):
        """check if the field exists"""
        field_found = False
        for col in self.db.get_columns(table_name):
            if col.name == field_name:
                field_found = True
                break

        if field_found:
            logger.info('Column %s exists in table %s' % (field_name, table_name))
        else:
            logger.warning('Column %s does not exists in table %s' % (field_name, table_name))
            if ddl_str is None:
                return False
            self.db.execute_sql(ddl_str)

        return True

    def delete_model(self, table_name, model_class):
        """Delete the table relate to the model"""
        if self.db.table_exists(table_name):
            self.db.drop_tables([model_class])
        else:
            logger.warning('Table %s not found, cannot delete it' % (table_name,))

    def delete_field(self, table_name, field_name):
        """Delete a field on a database"""
        query = "ALTER TABLE %s DROP COLUMN %s;" % (table_name, field_name)
        self.db.execute_sql(query)
        return True

    def add_service(self, wardcode, wardname):
        """Check if DEENOVA appear on f_dest"""
        try:
            Service.get(code=wardcode)
        except DoesNotExist:
            ser = Service()
            ser.code = wardcode
            ser.libelle = wardname
            ser.save()
        return True
