from peewee import (
    IntegerField,
    CharField,
    DateTimeField,
    BooleanField,
    AutoField,
    FloatField,
    TextField
)
from peewee import SQL, DoesNotExist, fn
from playhouse.hybrid import hybrid_method
from ..base import BaseModel
from .stock import Stock
from .magasin import Zone, ZoneMag
from .dispensation import FItem, FListe
import datetime

FRACTION = [
    25, 50,
    75, 100
]


class Product(BaseModel):

    pk = AutoField(column_name="x_pk", primary_key=True)
    reference = CharField(
        column_name="x_ref", unique=True, null=True, constraints=[SQL("DEFAULT ''")]
    )
    designation = CharField(
        column_name="x_desig", null=True, constraints=[SQL("DEFAULT ''")]
    )
    tmodif = DateTimeField(
        column_name="x_tmodif",
        null=True,
        constraints=[SQL("DEFAULT '0000-00-00 00:00:00'")],
    )
    tsortie = DateTimeField(
        column_name="x_tsortie",
        null=True,
        constraints=[SQL("DEFAULT '0000-00-00 00:00:00'")],
    )
    tentree = DateTimeField(
        column_name="x_tentree",
        null=True,
        constraints=[SQL("DEFAULT '0000-00-00 00:00:00'")],
    )
    cip = CharField(column_name="x_cip", null=True, constraints=[SQL("DEFAULT ''")])
    ucd = CharField(column_name="x_ucd", null=True, constraints=[SQL("DEFAULT ''")])
    dci = CharField(column_name="x_dci", constraints=[SQL("DEFAULT ''")])
    forme = CharField(column_name="x_forme", constraints=[SQL("DEFAULT ''")])
    desig_bis = CharField(column_name="x_desig_bis", constraints=[SQL("DEFAULT ''")])
    # unit_gest = CharField(column_name="x_unit_gest", constraints=[SQL("DEFAULT ''")])
    # unit_admin = CharField(column_name="x_unit_admin", constraints=[SQL("DEFAULT ''")])
    coef_conv = FloatField(
        column_name="x_coef_conv", constraints=[SQL("DEFAULT 0.000")]
    )
    delai_peremp = IntegerField(
        column_name="x_delai_peremp", constraints=[SQL("DEFAULT 0")]
    )
    externe = BooleanField(
        column_name="x_externe", null=True, constraints=[SQL("DEFAULT 0")]
    )
    conversion_ucd = BooleanField(
        column_name="x_conversion_ucd", null=True, constraints=[SQL("DEFAULT 0")]
    )
    multidose_par_uf = BooleanField(
        column_name="x_multidose_par_uf", constraints=[SQL("DEFAULT 0")]
    )
    com_med = TextField(
        column_name='x_com_med', null=True)
    bac = IntegerField(
        column_name='x_bac', null=True)

    @hybrid_method
    def stock_liste(self, typemag, fraction):
        ret = 0
        try:
            stk = FItem().select(
                (fn.SUM(FItem.qte_dem)-fn.SUM(FItem.qte_serv)).alias("total")
            ).join(FListe, on=(FListe.liste == FItem.liste)).where(
                FListe.zone_fin == typemag, FListe.mode == 'E',
                FItem.reference == self.reference,
                FItem.fraction == fraction
            )
            if len(stk):
                ret = stk[0].total or 0
        except DoesNotExist:  # pragma: no cover
            pass
        return ret

    @hybrid_method
    def stock_magasin(self, magasin, fraction):
        """Retrieve the quantity in stock"""
        ret = 0
        try:
            stk = Stock().select(
                fn.SUM(Stock.quantite).alias("total")
            ).where(
                Stock.magasin == magasin, Stock.reference == self.reference,
                Stock.fraction == fraction,
                Stock.date_peremption > datetime.datetime.now()
            )
            if len(stk):
                ret = stk[0].total or 0
        except DoesNotExist:  # pragma: no cover
            ret = 0
        return ret

    @hybrid_method
    def stock_coupe(self, type_mag, fraction):
        """Retrieve the quantity in stock"""
        ret = 0
        try:
            stk = Stock().select(fn.SUM(Stock.quantite).alias("total")).join(
                ZoneMag, on=(Stock.magasin == ZoneMag.mag)
            ).join(
                Zone, on=(Zone.zone == ZoneMag.zone & Zone.appro == 1)
            ).where(
                Stock.zone_admin == type_mag,
                Stock.reference == self.reference,
                Stock.fraction == fraction
            )
            if len(stk):
                ret = stk[0].total or 0
        except DoesNotExist:  # pragma: no cover
            ret = 0
        return ret

    class Meta:
        table_name = "f_ref"


class Seuil(BaseModel):

    pk = AutoField(
        column_name='x_pk', primary_key=True)
    reference = CharField(
        column_name='x_ref', constraints=[SQL("DEFAULT ''")])
    zone = CharField(
        column_name='x_zone', constraints=[SQL("DEFAULT ''")])
    stock_mini = FloatField(
        column_name='x_stk_mini', constraints=[SQL("DEFAULT 0.000")])
    stock_maxi = FloatField(
        column_name='x_stk_maxi', constraints=[SQL("DEFAULT 0.000")])
    fraction = IntegerField(
        column_name='x_fraction', constraints=[SQL("DEFAULT 100")])

    class Meta:
        table_name = 'f_seuil'

    def save(self, force_insert=False, only=None):
        """Enforce control on each information passed"""
        if self.fraction not in FRACTION:
            raise ValueError('Fraction value incorrect')
        return super().save(force_insert, only)


class LotRetire(BaseModel):

    pk = AutoField(
        column_name='x_pk', primary_key=True)
    ucd = CharField(
        column_name='x_ucd', constraints=[SQL("DEFAULT ''")])
    lot = CharField(
        column_name='x_lot', constraints=[SQL("DEFAULT ''")])

    class Meta:
        table_name = 'f_lot_retire'


class TopFive(BaseModel):

    pk = AutoField(
        column_name='x_pk', primary_key=True)
    cle = CharField(
        column_name='x_cle', max_length=10,
        constraints=[SQL("DEFAULT ''")])
    val = CharField(
        column_name='x_val', max_length=20,
        constraints=[SQL("DEFAULT ''")])
    poste = CharField(
        column_name='x_poste', max_length=20,
        constraints=[SQL("DEFAULT ''")])
    user = CharField(
        column_name='x_user', max_length=20,
        constraints=[SQL("DEFAULT ''")])
    mode = BooleanField(
        column_name='x_mode', constraints=[SQL("DEFAULT 0")])
    nb = IntegerField(
        column_name='x_nb', null=True, constraints=[SQL("DEFAULT 0")])

    class Meta:
        table_name = 'f_top5'
        indexes = (
            (('x_cle', 'x_val', 'x_poste', 'x_mode'), True),
        )
