# flake8: noqa
import pytest
from peewee import DoesNotExist
from median.models import Patient, Prescription, Dispensation, Service
import time
from datetime import datetime

@pytest.yield_fixture()
def rectest():
    print("Running method level setUp")
    rec_data = {}

    pat_test = Patient(nom='Test', prenom='Patrick')
    pat_test.ipp = "ABCD1234"
    pat_test.sexe = "M"
    pat_test.date_maj = datetime.now()
    pat_test.save()

    presc = Prescription()
    presc.ordre = 'Prescription 1'
    presc.ipp = 'ECO1234'
    presc.sejour = 'SEJ1234'
    presc.nom = "Test"
    presc.prenom = "Utilsiateur"
    presc.save()

    serv = Service()
    serv.code = "ECOTEST"
    serv.libelle = "Service de test"
    serv.ordre = 0
    serv.type_dest = "SERVICE"
    serv.moment_deb = "matin"
    serv.deb_matin = 7
    serv.deb_midi = 12
    serv.deb_soir = 18
    serv.deb_coucher = 21
    serv.nb_jour = 7
    serv.save()

    rec_data['patient'] = pat_test
    rec_data['prescription'] = presc
    rec_data['service'] = serv
    yield rec_data

    print("Running method level tearDown")
    pat_test.delete_instance()
    presc.delete_instance()
    serv.delete_instance()


class TestModelDispensation:

    def test_unknown_dispensation(self):
        """Test access unknown dispensation"""
        with pytest.raises(DoesNotExist):
            Dispensation.get(liste='INCONNU')

    def test_dispensation_out_wo_prescription(self, rectest):
        """Test create a dispensation without precription"""
        with pytest.raises(ValueError):
            dis = Dispensation()
            dis.liste = "TEST dispensation"
            dis.mode = 'S'
            dis.etat = 'V'
            dis.type_servi = 'NOMINATIF'
            dis.save()

    def test_dispensation_out_wo_prescription_exotique(self, rectest):
        """Test create a dispensation without precription, exoticque mode"""
        with pytest.raises(ValueError):
            dis = Dispensation()
            dis.liste = "TEST dispensation"
            dis.mode = 'S'
            dis.etat = 'V'
            dis.type_servi = 'EXOTIQUE'
            dis.save()

    def test_dispensation_out_wo_patient(self, rectest):
        """Test create a dispensation without patient"""
        with pytest.raises(ValueError):
            dis = Dispensation()
            dis.liste = "TEST dispensation"
            dis.mode = 'S'
            dis.etat = 'V'
            dis.type_servi = 'NOMINATIF'
            dis.id_prescription = rectest['prescription']
            dis.save()

    def test_create_dispensation_out(self, rectest):
        """Test create a dispensation"""
        dis = Dispensation()
        dis.liste = "TEST dispensation"
        dis.mode = 'S'
        dis.etat = 'V'
        dis.type_servi="NOMINATIF"
        dis.id_prescription = rectest['patient']
        dis.ipp = rectest['patient']
        dis.save()
        assert dis.pk > 0
        dis.delete_instance()

    def test_create_bad_output_lot(self, rectest):
        """Test create an output lot list"""
        with pytest.raises(ValueError):
            dis = Dispensation()
            dis.liste = "LOT A SORTIR xxx"
            dis.fusion = "SORTIE DE LOTS"
            dis.service = rectest['service'].code
            dis.mode = 'S'
            dis.etat = 'V'
            dis.save()

    def test_create_output_lot(self, rectest):
        """Test create an output lot list"""
        dis = Dispensation()
        dis.liste = "LOT A SORTIR xxx"
        dis.fusion = "SORTIE DE LOTS"
        dis.service = rectest['service'].code
        dis.type_servi="GLOBAL_BTE"
        dis.id_servi = 2
        dis.mode = 'S'
        dis.etat = 'V'
        dis.save()
        assert dis.pk > 0
        dis.delete_instance()

    def test_create_output_lot_wo_state(self, rectest):
        """Test create an output lot list without state"""
        dis = Dispensation()
        dis.liste = "LOT A SORTIR xxx"
        dis.fusion = "SORTIE DE LOTS"
        dis.service = rectest['service'].code
        dis.type_servi="GLOBAL_BTE"
        dis.id_servi = 2
        dis.mode = 'S'
        dis.save()
        assert dis.pk > 0
        dis.delete_instance()

    def test_output_expiry_wo_type_servi_erreur(self, rectest):
        """Test bad output of product expiry"""
        with pytest.raises(ValueError):
            dis = Dispensation()
            dis.liste = "PERIME A SORTIR xxx"
            dis.fusion = "SORTIE DE PERIMES"
            dis.service = rectest['service'].code
            dis.mode = 'S'
            dis.etat = 'V'
            dis.save()

    def test_output_expiry_error_wo_state(self, rectest):
        """Test bad output of product expiry"""
        dis = Dispensation()
        dis.liste = "PERIME A SORTIR xxx"
        dis.fusion = "SORTIE DE PERIMES"
        dis.service = rectest['service'].code
        dis.type_servi="GLOBAL_BTE"
        dis.id_servi = 2
        dis.mode = 'S'
        dis.save()
        assert dis.pk > 0
        dis.delete_instance()
