from ..base import BaseView, BaseViewException
from median.models import Product, Magasin, Gpao, FListe, FItem, Historique
from median.models import Seuil as modSeuil
from median.views import RawConfig
from median.utils import logger
from peewee import DoesNotExist
from functools import reduce
from median.database import mysql_db
import datetime
import operator
import time


class Seuil(BaseView):
    """Gestion des seuils dans Median"""

    def __init__(self, prod=None, mag=None):
        """Initialise le seuil"""
        self.produit = prod
        self.magasin = mag
        self.fraction = 100

    def _check_produit(self, produit):
        """Check if product exists

        :param produit: code produit a rechercher
        :type  produit: str
        :return:  Project model
        :rtype: median.models.Product
        """
        try:
            p = Product.get(reference=produit)
            return p
        except DoesNotExist as e:
            logger.error("Le produit %s n'existe pas" % produit)
            raise BaseViewException("Le produit n'existe pas dans la base")

    def _check_magasin(self, magasin):
        """Check if magasin exists

        :param magasin: code magasin a rechercher
        :type  magasin: str
        :return:  Magasin model
        :rtype: median.models.Magasin
        """
        try:
            m = Magasin.get(mag=magasin)
            return m
        except DoesNotExist as e:
            logger.error("Le magasin %s n'existe pas" % magasin)
            raise BaseViewException("Le magasin n'existe pas dans la base")

    def _check_type_magasin(self, type_magas):
        """Check if magasin exists

        :param type_magas: code magasin a rechercher
        :type  type_magas: str
        :return: Magasin model
        :rtype: median.models.Magasin
        """
        try:
            m = Magasin.get(type_mag=type_magas)
        except DoesNotExist as e:
            logger.error("Le type magasin %s n'existe pas" % type_magas)
            raise BaseViewException("Le magasin n'existe pas dans la base")
        return m

    def _transfert_service(self,):
        """Retrieve teh code of the service to transfert stock"""
        service = RawConfig().read(param='k_ua_transfert').value
        if service is None:
            raise BaseViewException("k_ua_transfet n'est pas définit")

        return service

    def liste(self, produit=None, magasin=None):
        """
        Affiche la liste des seuils d'un produit ou d'un magasin

        :param produit: code du produit a rechercher
        :type  produit: str
        :param magasin: Code du magasin
        :type  magasin: str
        :return: liste des seuils et de leur valeur
        :rtype: list
        """
        if produit is None and magasin is None:
            raise BaseViewException("Un produit ou magasin est nécessaire pour afficher la liste")

        clauses = []
        # vérification si le produit existe
        if produit is not None:
            try:
                p = Product.get(reference=produit)
                clauses.append((modSeuil.reference ==  produit))
            except DoesNotExist as e:
                logger.error("Le produit %s n'existe pas" % produit)
                raise BaseViewException("Le produit n'existe pas dans la base")

        # vérification si le magasin existe
        if magasin is not None:
            try:
                m = Magasin.get(mag=magasin)
                clauses.append((modSeuil.zone == m.type_mag))
            except DoesNotExist as e:
                logger.error("LE magasin %s n'existe pas" % magasin)
                raise BaseViewException("Le magasin n'existe pas dans la base")

        res = []
        s = modSeuil.select().where(reduce(operator.and_, clauses))

        for l in s:
            res.append({
                'pk': l.pk,
                'reference': l.reference,
                'magasin': l.zone,
                'mini': l.stock_mini,
                'maxi': l.stock_maxi,
                'fraction': l.fraction
            })
        return res

    def reappro(self, produit=None, magasin=None, quantite=0, fraction=100):
        """Reappro ligne de produit"""
        self.mandatory(produit, "Le produit est obligatoire")
        self.mandatory(magasin, "Le magasin est obligatoire")
        self.non_zero(quantite, "La quantité doit être supérieure à zéro")
        pro = self._check_produit(produit)
        mag = self._check_magasin(magasin)

        liste = '%s Unitaire LE %s' % (mag.type_mag, time.strftime('%Y-%m-%d'))
        service = self._transfert_service()

        # Check if liste is already exists
        try:
            d = FListe.get(liste=liste)
            d.nb_item = d.nb_item + 1
        except DoesNotExist:
            d = FListe(
                liste=liste,
                date_creation=datetime.datetime.now(),
                mode='E',
                etat='V',
                fusion='REASSORT',
                service=service,
                nb_item=1,
                date_modification=datetime.datetime.now(),
                zone_fin=mag.type_mag,
                type_servi='GLOBAL_BTE',
                id_servi=2,
            )
            d.save()

        # Create f_item
        itm = FItem(
            liste=liste,
            reference=produit,
            mode='E',
            etat='V',
            item=str(d.nb_item).zfill(6),
            qte_dem=quantite,
            fraction=fraction,
            service=service,
            type_servi='GLOBAL_BTE',
            id_servi=2,
        )
        itm.save()

        d.date_modification=datetime.datetime.now()
        d.save()

        # Create f_gpao
        # g = Gpao(
        #     chrono=datetime.datetime.now(),
        #     poste='MEDIANWEB',
        #     etat='A',
        #     type_mvt='R',
        #     ref=produit,
        #     qte=quantite,
        #     fraction=fraction,
        #     id_zone=mag.id_zone,
        #     liste=liste,
        #     item=str(d.nb_item).zfill(6),
        #     desig=pro.designation,
        # )
        # g.save()
        self.creer_gpao(mag.id_zone, produit, fraction)

        # Add trace on historique
        h = Historique(
            chrono=datetime.datetime.now(),
            reference=produit,
            type_mouvement='REF',
            liste=liste,
            info='Reappro %s, quantité %i, fraction %i' % (magasin, int(quantite), int(fraction)),
            service=service,
        )
        h.save()

        return liste

    # def reappro_liste(self, type_magasin=None, libelle=None):
    #     """Création d'une liste de produit

    #     :param type_magasin: code du type de magasin a réapprovisionner
    #     :type  type_magasin: string
    #     :return: la liste créée
    #     :rtype: FListe
    #     """
    #     mag = self._check_type_magasin(type_magasin)

    #     if libelle is None:
    #         liste = '%s WEB LE %s' % (mag.type_mag, time.strftime('%d/%m/%y'))
    #     else:
    #         liste = libelle

    #     # récupération de la liste de seuil du magasin
    #     l_seuil = modSeuil().select().where(modSeuil.zone==type_magasin)

    #     if not len(l_seuil):
    #         raise BaseViewException('Pas de seuil définit pour le magasin %s' % type_magasin)

    #     service = self._transfert_service()

    #     # Check if liste is already exists
    #     try:
    #         d = FListe.get(liste=liste)
    #     except DoesNotExist:
    #         d = FListe(
    #             liste=liste,
    #             date_creation=datetime.datetime.now(),
    #             mode='E',
    #             etat='V',
    #             fusion='REASSORT',
    #             service=service,
    #             nb_item=0,
    #             date_modification=datetime.datetime.now(),
    #             zone_fin=mag.type_mag,
    #             type_servi='GLOBAL_BTE',
    #             id_servi=2,
    #         )
    #         d.save()

    #     cpt = 0
    #     for ls in l_seuil:
    #         try:
    #             pro = Product.get(Product.reference==ls.reference)
    #         except Exception:
    #             continue

    #         quantite = ls.stock_maxi
    #         produit = ls.reference
    #         fraction = ls.fraction
    #         stk_liste = pro.stock_liste(mag.type_mag)
    #         stk_mag = pro.stock_magasin(mag.mag)

    #         quantite = quantite - stk_mag - stk_liste
    #         if quantite < 0:
    #             continue

    #         cpt += 1
    #         # Create f_item
    #         itm = FItem(
    #             liste=liste,
    #             reference=produit,
    #             mode='E',
    #             etat='V',
    #             item=str(cpt).zfill(6),
    #             qte_dem=quantite,
    #             fraction=fraction,
    #             service=service,
    #             type_servi='GLOBAL_BTE',
    #             id_servi=2,
    #         )
    #         itm.save()

    #         # Create f_gpao
    #         g = Gpao(
    #             chrono=datetime.datetime.now(),
    #             poste='MEDIANWEB',
    #             etat='A',
    #             type_mvt='R',
    #             ref=produit,
    #             qte=(quantite * fraction / 100),
    #             fraction=fraction,
    #             id_zone=mag.id_zone,
    #             id_robot=mag.id_robot,
    #             num_mag=mag.mag,
    #             liste=liste,
    #             item=str(cpt).zfill(6),
    #             desig=pro.designation,
    #         )
    #         g.save()

    #         # Add trace on historique
    #         h = Historique(
    #             chrono=datetime.datetime.now(),
    #             reference=produit,
    #             type_mouvement='REF',
    #             liste=liste,
    #             info='Reappro %s, quantité %i, fraction %i' % (mag.mag, quantite, fraction),
    #             service=service,
    #         )
    #         h.save()

    #     d.date_modification=datetime.datetime.now()
    #     d.nb_item = cpt
    #     d.save()
    #     return d


    def reappro_item(self, liste, item, ref, fraction, quantite, type_magasin, service, id_zone):
        """Création d'une liste de produit

        :param type_magasin: code du type de magasin a réapprovisionner
        :type  type_magasin: string
        :return: la liste créée
        :rtype: FListe
        """

        # service = self._transfert_service()

        # Create f_item
        itm = FItem(
            liste=liste,
            reference=ref,
            mode='E',
            etat='V',
            item=str(item).zfill(6),
            qte_dem=quantite,
            fraction=fraction,
            dest=service,
            type_servi='GLOBAL_BTE',
            id_servi=2,
        )
        itm.save()

        # Add trace on historique
        h = Historique(
            chrono=datetime.datetime.now(),
            reference=ref,
            type_mouvement='REF',
            liste=liste,
            info='Reappro %s, quantité %s, fraction %s' % (type_magasin, quantite, fraction),
            service=service,
        )
        h.save()

        # Create f_gpao
        # g = Gpao(
        #     chrono=datetime.datetime.now(),
        #     poste='MEDIANWEB',
        #     etat='A',
        #     type_mvt='R',
        #     ref=ref,
        #     qte=quantite,
        #     fraction=fraction,
        #     id_zone=id_zone,
        #     liste=liste,
        #     item=str(item).zfill(6)
        # )
        # g.save()

        return "ok"

    def reappro_calcul(self, type_magasin=None):
        """Compute all product to be refill"""
        res = []
        mag = self._check_type_magasin(type_magasin)
        service = self._transfert_service()
        l_seuil = modSeuil().select().where(modSeuil.zone==type_magasin)

        cpt = 0
        for ls in l_seuil:
            try:
                pro = Product.get(Product.reference==ls.reference)
            except Exception:
                continue

            quantite = ls.stock_maxi
            produit = ls.reference
            fraction = ls.fraction
            stk_liste = pro.stock_liste(mag.type_mag, fraction)
            stk_mag = pro.stock_magasin(mag.mag, fraction)
            stk_coupe = pro.stock_coupe(mag.type_mag, fraction)

            quantite = quantite - stk_mag - stk_liste - stk_coupe
            if quantite <= 0:
                continue

            # si on ne depasse pas le seuil mini, on passe au seuil suivant
            if (stk_mag + stk_liste + stk_coupe) >= ls.stock_mini:
                continue

            res.append({
                'checked': False,
                'ref': ls.reference,
                'name': pro.designation,
                'stkmini': ls.stock_mini,
                'stkmaxi': ls.stock_maxi,
                'fraction': ls.fraction,
                'quantite': quantite,
                'stock': stk_mag + stk_coupe,
                'commande': stk_liste
            })
        return res

    def creer_gpao(self, id_zone, ref=None, fract=100):
        extra = ''
        if ref is not None:
            extra = "AND s.x_ref='%s' AND s.x_fraction=%i" % (ref, fract)
        query = """
            insert into f_gpao (x_chrono, x_poste, x_ref, x_fraction, x_etat, x_qte, x_type_mvt, x_id_zone)
            SELECT NOW(), 'MEDIANWEB', x_ref, IFNULL(s.x_fraction,100), 'A', IFNULL((
            SELECT SUM(x_qte_dem-x_qte_serv)
            FROM f_item i
            INNER JOIN f_liste l ON l.x_liste = i.x_liste
            INNER JOIN f_mag m ON m.x_type_mag=l.x_zone_fin
            WHERE i. x_etat!='S' AND i.x_mode='E' AND i.x_ref=s.x_ref AND m.x_id_zone=m1.x_id_zone
            GROUP BY i.x_ref, x_fraction),0), 'R', x_id_zone
            FROM f_seuil s
            INNER JOIN f_mag m1 ON s.x_zone=m1.x_type_mag
            WHERE m1.x_id_zone={0}
            {1}
            GROUP BY x_id_zone, x_ref, x_fraction
            """.format(id_zone, extra)
        cr = mysql_db.execute_sql(query)

        return "ok"
