from peewee import DoesNotExist
from .utils import logger
from .models import (
    PrescriptionItem, MagasinService, LogAstus, TopFive,
    HistoriqueMultidoses, MessageAstus, FListeError,
    Service
)


class DbMigrator:

    def __init__(self, db_instance):
        """Initilise the migraotr class with the database selected"""
        self.db = db_instance
        logger.debug('Initilize migrator instance')

    def migrate(self):
        """Launch the migration script"""
        self.check_model('f_prescription_item', PrescriptionItem)
        self.check_model('f_mag_dest', MagasinService)
        self.check_model('f_log_astus', LogAstus)
        self.check_model('f_top5', TopFive)
        self.check_model('f_histo_multidoses', HistoriqueMultidoses)
        self.check_model('f_message_astus', MessageAstus)
        self.check_model('f_liste_error', FListeError)
        self.check_field('f_user', 'x_login', "ALTER TABLE f_user ADD COLUMN x_login VARCHAR(30) DEFAULT '';")
        self.check_field('f_user', 'x_email', "ALTER TABLE f_user ADD COLUMN x_email VARCHAR(320) DEFAULT '';")
        self.check_field('f_user', 'x_import_flag', "ALTER TABLE f_user ADD COLUMN x_import_flag TINYINT(1) NULL DEFAULT 0;")
        self.check_field('f_user', 'x_maintenance', "ALTER TABLE f_user ADD COLUMN x_maintenance TINYINT(1) NULL DEFAULT 0;")
        self.check_field('f_profil', 'x_visu', "ALTER TABLE f_profil ADD COLUMN x_visu TINYINT(1) NULL DEFAULT 0;")
        self.check_field('f_profil', 'x_edit', "ALTER TABLE f_profil ADD COLUMN x_edit TINYINT(1) NULL DEFAULT 0;")
        self.check_field('f_ref', 'x_com_med', "ALTER TABLE f_ref ADD COLUMN x_com_med TEXT DEFAULT '';")
        self.check_field('f_ref', 'x_bac', "ALTER TABLE f_ref ADD COLUMN x_bac INT(11) NULL DEFAULT 0;")
        self.add_service('DEENOVA', 'Service de test')


    def check_model(self, table_name, model_class):
        """Check if table exists, and create it if missing"""
        if not self.db.table_exists(table_name):
            logger.warning('Table %s does not exists, we create it from %s' % (table_name, model_class))
            self.db.create_tables([model_class])
        else:
            logger.info('Table %s exists' % (table_name,))

        return True

    def check_field(self, table_name, field_name, ddl_str=None):
        """check if the field exists"""
        field_found = False
        for col in self.db.get_columns(table_name):
            if col.name == field_name:
                field_found = True
                break

        if field_found:
            logger.info('Column %s exists in table %s' % (field_name, table_name))
        else:
            logger.warning('Column %s does not exists in table %s' % (field_name, table_name))
            if ddl_str is None:
                return False
            self.db.execute_sql(ddl_str)

        return True

    def delete_model(self, table_name, model_class):
        """Delete the table relate to the model"""
        if self.db.table_exists(table_name):
            self.db.drop_tables([model_class])
        else:
            logger.warning('Table %s not found, cannot delete it' % (table_name,))

    def delete_field(self, table_name, field_name):
        """Delete a field on a database"""
        query = "ALTER TABLE %s DROP COLUMN %s;" % (table_name, field_name)
        self.db.execute_sql(query)
        return True

    def add_service(self, wardcode, wardname):
        """Check if DEENOVA appear on f_dest"""
        try:
            Service.get(code=wardcode)
        except DoesNotExist:
            ser = Service()
            ser.code = wardcode
            ser.libelle = wardname
            ser.save()
        return True
