# flake8: noqa
import pytest
from peewee import DoesNotExist, IntegrityError
from median.models import Product, Ucd, Cip, GtinAlias, Seuil, ContainerFormat, GtinDimension
from median.base import MedianLibException


class TestModelProduct:
    def test_unknown_product(self):
        """Test d'un produit inexistant"""
        with pytest.raises(DoesNotExist):
            Product.get(reference="INCONNU")

    def test_select_product(self):
        """Verification du produit deverminage"""
        pr = Product.get(reference="DEVERMINAGE")
        assert pr.pk > 0

    def test_create_product(self):
        """Creation d'un produit de test"""
        pr = Product()
        pr.reference = "ECO1234"
        pr.designation = "Eco Medoc"
        pr.save()
        assert pr.pk > 0

    def test_modify_product(self):
        """Modify product (eg: astus only)"""
        pr = Product.get(reference="ECO1234")
        assert pr.externe == False
        pr.externe = True
        pr.save()
        assert pr.externe == True

    def test_delete_product(self):
        """Suppression du produitd e test"""
        pr = Product.get(reference="ECO1234")
        pr.delete_instance()

    def test_create_product_with_known_anomaly(self):
        """Creation d'un produit de test avec anomalie connue"""
        pr = Product()
        pr.reference = "ECO1234"
        pr.designation = "Eco Medoc"
        pr.anomaly = "UCD_DBL"
        pr.save()
        assert pr.pk > 0
        pr.delete_instance()

    def test_create_product_with_unknown_anomaly(self):
        """Creation d'un produit de test avec anomalie connue"""
        with pytest.raises(ValueError):
            pr = Product()
            pr.reference = "ECO1234"
            pr.designation = "Eco Medoc"
            pr.anomaly = "UNKNOWN"
            pr.save()

    def test_create_product_ucd(self):
        """Creation d'un produit de test avec UCD"""
        pr = Product()
        pr.reference = "ECO9999"
        pr.designation = "Eco Medoc"
        pr.save()
        assert pr.pk > 0
        ucd = Ucd()
        ucd.reference = pr.reference
        ucd.ucd = "9047693"
        ucd.save()
        assert ucd.pk > 0
        ucd.delete_instance()
        pr.delete_instance()

    def test_product_list_ucd(self):
        """Lister les UCD d'un produit"""
        pr = Product()
        pr.reference = "ECO9999A"
        pr.designation = "Eco Medoc"
        pr.save()
        assert pr.pk > 0
        ucd = Ucd()
        ucd.reference = pr.reference
        ucd.ucd = "9047664"
        ucd.save()
        assert ucd.pk > 0
        prt = Product.get(reference=pr.reference)
        assert len(prt.cdu_list()) == 1
        ucd.delete_instance()
        pr.delete_instance()

    def test_product_list_two_ucd(self):
        """Lister les UCD d'un produit"""
        pr = Product()
        pr.reference = "ECO9999A"
        pr.designation = "Eco Medoc"
        pr.save()
        assert pr.pk > 0
        ucd = Ucd()
        ucd.reference = pr.reference
        ucd.ucd = "9047664"
        ucd.save()
        assert ucd.pk > 0
        ucd2 = Ucd()
        ucd2.reference = pr.reference
        ucd2.ucd = "9054664"
        ucd2.save()
        assert ucd2.pk > 0
        prt = Product.get(reference=pr.reference)
        assert len(prt.cdu_list()) == 2
        ucd2.delete_instance()
        ucd.delete_instance()
        pr.delete_instance()

    def test_create_product_ucd_cip(self):
        """Creation d'un produit de test avec UCD et CIP"""
        pr = Product()
        pr.reference = "ECO9999"
        pr.designation = "Eco Medoc"
        pr.save()
        assert pr.pk > 0
        ucd = Ucd()
        ucd.reference = pr.reference
        ucd.ucd = "9047693"
        ucd.save()
        assert ucd.pk > 0
        cip = Cip()
        cip.ucd = ucd.ucd
        cip.cip = "3400937140423"
        cip.save()
        assert cip.pk > 0
        assert cip.dossier == "0"
        cip.delete_instance()
        ucd.delete_instance()
        pr.delete_instance()

    def test_product_ucd_gtin_list(self):
        """Creation liste CIP"""
        pr = Product()
        pr.reference = "ECO9999A"
        pr.designation = "Eco Medoc"
        pr.save()
        assert pr.pk > 0
        ucd = Ucd()
        ucd.reference = pr.reference
        ucd.ucd = "9147693"
        ucd.save()
        assert ucd.pk > 0
        cip = Cip()
        cip.ucd = ucd.ucd
        cip.cip = "3400937240423"
        cip.save()
        assert cip.pk > 0
        assert cip.dossier == "0"
        prt = Product.get(reference=pr.reference)
        assert len(prt.gtin_list()) == 1
        cip.delete_instance()
        ucd.delete_instance()
        pr.delete_instance()

    def test_product_ucd_gtin_list_two(self):
        """Creation d'un produit de test avec UCD et CIP"""
        pr = Product()
        pr.reference = "ECO9999A"
        pr.designation = "Eco Medoc"
        pr.save()
        assert pr.pk > 0
        ucd = Ucd()
        ucd.reference = pr.reference
        ucd.ucd = "9147693"
        ucd.save()
        assert ucd.pk > 0
        cip = Cip()
        cip.ucd = ucd.ucd
        cip.cip = "3400937240423"
        cip.save()
        assert cip.pk > 0
        assert cip.dossier == "0"
        cip2 = Cip()
        cip2.ucd = ucd.ucd
        cip2.cip = "3400937340423"
        cip2.save()
        assert cip2.pk > 0
        prt = Product.get(reference=pr.reference)
        assert len(prt.gtin_list()) == 2
        cip2.delete_instance()
        cip.delete_instance()
        ucd.delete_instance()
        pr.delete_instance()

    def test_gtin_alias_creation(self):
        pr = Product()
        pr.reference = "ECO9999"
        pr.designation = "Eco Medoc"
        pr.save()
        assert pr.pk > 0
        ucd = Ucd()
        ucd.reference = pr.reference
        ucd.ucd = "9047693"
        ucd.save()
        assert ucd.pk > 0
        cip = Cip()
        cip.ucd = ucd.ucd
        cip.cip = "3400937140423"
        cip.save()
        assert cip.pk > 0
        assert cip.dossier == "0"
        alias = GtinAlias()
        alias.source = "25500100000012"
        alias.destination = "3400937140423"
        alias.save()
        assert alias.pk > 0
        alias.delete_instance()
        cip.delete_instance()
        ucd.delete_instance()
        pr.delete_instance()

    def test_create_gtin_dimension(self):
        gdim = GtinDimension()
        gdim.code = "3215468796585"
        gdim.witdh = 10
        gdim.height = 10
        gdim.depth = 10
        gdim.save()

        assert gdim.volume() == 1000

        gdim.delete_instance()

    def test_create_seuil_fraction_100_ok(self):
        """Creation d'un seuil pour la fraction 100"""
        s = Seuil()
        s.reference = "4556485"
        s.zone = "CUEILLETTE_100"
        s.stock_mini = 14
        s.stock_maxi = 28
        s.fraction = 100
        s.save()
        assert s.pk > 0
        s.delete_instance()

    def test_create_seuil_fraction_50_ok(self):
        """Creation d'un seuil pour la fraction 30"""
        s = Seuil()
        s.reference = "4556485"
        s.zone = "CUEILLETTE_100"
        s.stock_mini = 14
        s.stock_maxi = 28
        s.fraction = 50
        s.save()
        assert s.pk > 0
        s.delete_instance()

    def test_create_seuil_fraction_30_fail(self):
        """interdire la creation d'un seuil pour la fraction 30"""
        with pytest.raises(ValueError):
            s = Seuil()
            s.reference = "4556485"
            s.zone = "CUEILLETTE_100"
            s.stock_mini = 15
            s.stock_maxi = 30
            s.fraction = 30
            s.save()

    def test_product_stock_unknown_mag(self):
        """Test produit sur magasin existant"""
        pr = Product()
        pr.reference = "ECO1234"
        pr.designation = "Eco Medoc"
        pr.save()
        assert pr.pk > 0

        ret = pr.stock_magasin("UNKNOWN", 100)
        assert ret == 0

        pr.delete_instance()

    def test_container_format_creation(self):
        """Create a container format"""
        cf = ContainerFormat()
        cf.code = "BACTEST"
        cf.save()

        cf.delete_instance()

    def test_container_format_creation_twice(self):
        """Create a container format twice"""
        cf = ContainerFormat()
        cf.code = "BACTEST"
        cf.save()

        with pytest.raises(IntegrityError):
            cf2 = ContainerFormat()
            cf2.code = "BACTEST"
            cf2.save()

        cf.delete_instance()

    def test_container_format_dimension_x_negative(self):
        """Create a container format with negaitve Y dimension"""
        with pytest.raises(MedianLibException) as e:
            cf = ContainerFormat()
            cf.code = "BACDIMX"
            cf.dimension_x = -1
            cf.save()
        assert "Width must greather or equal to 0" in str(e.value)

    def test_container_format_dimension_y_negative(self):
        """Create a container format with negaitve Y dimension"""
        with pytest.raises(MedianLibException) as e:
            cf = ContainerFormat()
            cf.code = "BACDIMY"
            cf.dimension_y = -1
            cf.save()
        assert "Height must greather or equal to 0" in str(e.value)

    def test_container_format_dimension_z_negative(self):
        """Create a container format with negaitve Z dimension"""
        with pytest.raises(MedianLibException) as e:
            cf = ContainerFormat()
            cf.code = "BACDIMZ"
            cf.dimension_z = -1
            cf.save()
        assert "Depth must greather or equal to 0" in str(e.value)

    def test_container_format_creation(self):
        """Create a container format"""
        cf = ContainerFormat()
        cf.code = "BACVOLU"
        cf.measurement_x_mm = 5
        cf.measurement_y_mm = 10
        cf.measurement_z_mm = 3
        cf.save()

        assert cf.volume() == 150

        cf.delete_instance()
