# flake8: noqa
import pytest
from peewee import DoesNotExist
from datetime import datetime
from median.base import BaseViewException, BaseViewDuplicate
from median.models import Product, Adresse, Stock
from median.views import Inventaire


@pytest.fixture()
def rectest():
    """Code commun à toute les fonctions de ce test"""
    rec_data = {}
    # Premier produit
    pr1 = Product()
    pr1.reference = "ECO6547"
    pr1.designation = "Eco Medoc"
    pr1.save()
    rec_data["product_1"] = pr1
    # Second produit
    pr2 = Product()
    pr2.reference = "ECO9876"
    pr2.designation = "Eco Medoc 2"
    pr2.save()
    rec_data["product_2"] = pr2
    # Emplacement 1
    adr1 = Adresse()
    adr1.adresse = "ST1.192.168.100.120"
    adr1.etat = "L"
    adr1.format = "BOITE PASS"
    adr1.magasin = "ST1"
    adr1.save()
    rec_data["adr_1"] = adr1
    # Emplacement 2
    adr2 = Adresse()
    adr2.adresse = "ST1.192.168.200.012"
    adr2.etat = "L"
    adr2.format = "BOITE PASS"
    adr2.magasin = "ST1"
    adr2.save()
    rec_data["adr_2"] = adr2
    # Emplacement 3
    adr3 = Adresse()
    adr3.adresse = "U01.212.168.200.012"
    adr3.etat = "L"
    adr3.format = "BAC 1/7 B"
    adr3.magasin = "U01"
    adr3.save()
    rec_data["adr_3"] = adr3
    # Emplacement 4
    adr4 = Adresse()
    adr4.adresse = "ST1.192.168.250.012"
    adr4.etat = "O"
    adr4.format = "BOITE PASS"
    adr4.magasin = "ST1"
    adr4.save()
    rec_data["adr_4"] = adr4
    # Emplacement 5 bloqué
    adr5 = Adresse()
    adr5.adresse = "ST1.192.168.250.050"
    adr5.etat = "X"
    adr5.format = "BOITE PASS"
    adr5.magasin = "ST1"
    adr5.save()
    rec_data["adr_5"] = adr5

    # Ajout de stock à l'emplace 4
    stk4 = Stock()
    stk4.reference = pr1.reference
    stk4.adresse = adr4.adresse
    stk4.lot = "ABCDS"
    stk4.quantite = 15
    stk4.date_entree = datetime.now()
    stk4.save()

    yield rec_data

    # function test finish, we clean the data
    pr1.delete_instance()
    pr2.delete_instance()
    adr1.delete_instance()
    adr2.delete_instance()
    adr3.delete_instance()
    adr4.delete_instance()
    adr5.delete_instance()
    stk4.delete_instance()


class TestObjectInventaire:
    def test_move_product_different_format(self, rectest):
        """Déplacement dans un emplacement qui n'a pas le même format"""
        with pytest.raises(BaseViewException):
            ret = Inventaire().deplacer_produit(
                rectest["adr_1"].adresse,
                rectest["adr_3"].adresse,
                rectest["product_1"].reference,
            )

    def test_move_product_location_occupied(self, rectest):
        """Déplacement dans un emplacement qui est occupé"""
        with pytest.raises(BaseViewException):
            ret = Inventaire().deplacer_produit(
                rectest["adr_1"].adresse,
                rectest["adr_4"].adresse,
                rectest["product_1"].reference,
            )

    def test_destination_missing_dest_location(self, rectest):
        """L'emplacement de destination n'existe pas"""
        with pytest.raises(BaseViewException) as e:
            ret = Inventaire().deplacer_produit(
                rectest["adr_1"].adresse,
                "U01.XXX.YYY.ZZZ",
                rectest["product_1"].reference,
            )
        assert "l'adresse destination n'existe pas" in str(e.value)

    def test_destination_missing_src_location(self, rectest):
        """L'emplacement source n'existe pas"""
        with pytest.raises(BaseViewException) as e:
            ret = Inventaire().deplacer_produit(
                "U01.XXX.YYY.ZZZ",
                rectest["adr_2"].adresse,
                rectest["product_1"].reference,
            )
        assert "l'adresse source n'existe pas" in str(e.value)

    def test_move_product_without_stock(self, rectest):
        """Pas  de stock sur l'emplacement source"""
        with pytest.raises(BaseViewException) as e:
            ret = Inventaire().deplacer_produit(
                rectest["adr_2"].adresse,
                rectest["adr_1"].adresse,
                rectest["product_1"].reference,
            )
        assert "Pas de stock à l'emplacement source" in str(e.value)

    def test_move_product_with_stock_without_product(self, rectest):
        """Du stock sur l'emplacement source, mais pas de produit spécifié"""
        with pytest.raises(BaseViewException) as e:
            ret = Inventaire().deplacer_produit(
                rectest["adr_4"].adresse, rectest["adr_1"].adresse
            )
        assert "Le produit est obligatoire" in str(e.value)

    def test_deplacement_produit_adr4_vers_adr2(self, rectest):
        """Deplacement du stock"""
        # ret = Inventaire().deplacer_produit
        pass

    def test_deplacement_produit_adr1_to_adr5(self, rectest):
        """Test move product to an block addresse"""
        with pytest.raises(BaseViewException) as e:
            ret = Inventaire().deplacer_produit(
                rectest["adr_1"].adresse, rectest["adr_5"].adresse
            )

        assert "emplacement de destination est bloqué" in str(e.value)

    def test_deplacement_produit_move_to_adr5(self, rectest):
        ret = Inventaire().deplacer_produit(
            rectest["adr_4"].adresse,
            rectest["adr_2"].adresse,
            rectest["product_1"].reference,
        )
