# flake8: noqa
import pytest
from peewee import DoesNotExist
from median.models import Patient, Prescription, Dispensation, Service, FListeError
from median.constant import PatientGlobal
from median.base import MedianLibException
import time
from datetime import datetime


@pytest.fixture()
def rectest():
    print("Running method level setUp")
    rec_data = {}

    pat_test = Patient(nom="Test", prenom="Patrick")
    pat_test.ipp = "ABCD1234"
    pat_test.sexe = "M"
    pat_test.date_maj = datetime.now()
    pat_test.save()

    presc = Prescription()
    presc.ordre = "Prescription 1"
    presc.ipp = "ECO1234"
    presc.sejour = "SEJ1234"
    presc.nom = "Test"
    presc.prenom = "Utilsiateur"
    presc.save()

    serv = Service()
    serv.code = "ECOTEST"
    serv.libelle = "Service de test"
    serv.ordre = 0
    serv.type_dest = "SERVICE"
    serv.moment_deb = "matin"
    serv.deb_matin = 7
    serv.deb_midi = 12
    serv.deb_soir = 18
    serv.deb_coucher = 21
    serv.nb_jour = 7
    serv.save()

    rec_data["patient"] = pat_test
    rec_data["prescription"] = presc
    rec_data["service"] = serv
    yield rec_data

    print("Running method level tearDown")
    pat_test.delete_instance()
    presc.delete_instance()
    serv.delete_instance()


class TestModelDispensation:
    def test_unknown_dispensation(self):
        """Test access unknown dispensation"""
        with pytest.raises(DoesNotExist):
            Dispensation.get(liste="INCONNU")

    def test_dispensation_out_wo_prescription(self, rectest):
        """Test create a dispensation without precription"""
        with pytest.raises(ValueError):
            dis = Dispensation()
            dis.liste = "TEST dispensation"
            dis.mode = "S"
            dis.etat = "V"
            dis.type_servi = "NOMINATIF"
            dis.save()

    def test_dispensation_out_wo_prescription_exotique(self, rectest):
        """Test create a dispensation without precription, exoticque mode"""
        with pytest.raises(ValueError):
            dis = Dispensation()
            dis.liste = "TEST dispensation"
            dis.mode = "S"
            dis.etat = "V"
            dis.type_servi = "EXOTIQUE"
            dis.save()

    def test_dispensation_out_wo_patient(self, rectest):
        """Test create a dispensation without patient"""
        with pytest.raises(ValueError):
            dis = Dispensation()
            dis.liste = "TEST dispensation"
            dis.mode = "S"
            dis.etat = "V"
            dis.type_servi = "NOMINATIF"
            dis.id_prescription = rectest["prescription"]
            dis.save()

    def test_create_dispensation_out(self, rectest):
        """Test create a dispensation"""
        dis = Dispensation()
        dis.liste = "TEST dispensation"
        dis.mode = "S"
        dis.etat = "V"
        dis.type_servi = "NOMINATIF"
        dis.id_prescription = rectest["prescription"].pk
        dis.num_ipp = rectest["patient"].ipp
        dis.num_sej = 'SEJ1234'
        dis.save()
        assert dis.pk > 0
        dis.delete_instance()

    def test_create_bad_output_lot(self, rectest):
        """Test create an output lot list"""
        with pytest.raises(ValueError):
            dis = Dispensation()
            dis.liste = "LOT A SORTIR xxx"
            dis.fusion = "SORTIE DE LOTS"
            dis.service = rectest["service"].code
            dis.mode = "S"
            dis.etat = "V"
            dis.save()

    def test_create_output_lot(self, rectest):
        """Test create an output lot list"""
        dis = Dispensation()
        dis.liste = "LOT A SORTIR xxx"
        dis.fusion = "SORTIE DE LOTS"
        dis.service = rectest["service"].code
        dis.type_servi = "GLOBAL_BTE"
        dis.num_ipp = PatientGlobal.Ipp.value
        dis.num_sej = PatientGlobal.Sejour.value
        dis.id_servi = 2
        dis.mode = "S"
        dis.etat = "V"
        dis.save()
        assert dis.pk > 0
        dis.delete_instance()

    def test_riedl_create_output_lot(self, rectest):
        """Test create an output lot list"""
        dis = Dispensation()
        dis.liste = "RIEDL LOT OUTPUT"
        dis.fusion = "SORTIE DE LOTS"
        dis.service = rectest["service"].code
        dis.type_servi = "RIEDL_BTE"
        dis.num_ipp = PatientGlobal.Ipp.value
        dis.num_sej = PatientGlobal.Sejour.value
        dis.id_servi = 2
        dis.mode = "S"
        dis.etat = "V"
        dis.save()
        assert dis.pk > 0
        assert dis.zone_deb == "RIEDL"
        dis.delete_instance()

    def test_create_output_lot_wo_state(self, rectest):
        """Test create an output lot list without state"""
        dis = Dispensation()
        dis.liste = "LOT A SORTIR xxx"
        dis.fusion = "SORTIE DE LOTS"
        dis.service = rectest["service"].code
        dis.type_servi = "GLOBAL_BTE"
        dis.num_ipp = PatientGlobal.Ipp.value
        dis.num_sej = PatientGlobal.Sejour.value
        dis.id_servi = 2
        dis.mode = "S"
        dis.save()
        assert dis.pk > 0
        dis.delete_instance()

    def test_output_expiry_wo_type_servi_erreur(self, rectest):
        """Test bad output of product expiry"""
        with pytest.raises(ValueError):
            dis = Dispensation()
            dis.liste = "PERIME A SORTIR xxx"
            dis.fusion = "SORTIE DE PERIMES"
            dis.service = rectest["service"].code
            dis.mode = "S"
            dis.etat = "V"
            dis.save()

    def test_global_output_wo_ipp(self, rectest):
        """Test bad output of product expiry"""
        with pytest.raises(ValueError):
            dis = Dispensation()
            dis.liste = "PERIME A SORTIR xxx"
            dis.fusion = "SORTIE DE PERIMES"
            dis.service = rectest["service"].code
            dis.type_servi = "GLOBAL_BTE"
            dis.id_servi = 2
            dis.mode = "S"
            dis.save()
            assert dis.pk > 0
            dis.delete_instance()

    def test_global_output_wo_episode(self, rectest):
        """Test bad output of product expiry"""
        with pytest.raises(ValueError):
            dis = Dispensation()
            dis.liste = "PERIME A SORTIR xxx"
            dis.fusion = "SORTIE DE PERIMES"
            dis.service = rectest["service"].code
            dis.type_servi = "GLOBAL_BTE"
            dis.num_ipp = PatientGlobal.Ipp.value
            dis.id_servi = 2
            dis.mode = "S"
            dis.save()
            assert dis.pk > 0
            dis.delete_instance()


    def test_output_expiry_error_wo_state(self, rectest):
        """Test bad output of product expiry"""
        dis = Dispensation()
        dis.liste = "PERIME A SORTIR xxx"
        dis.fusion = "SORTIE DE PERIMES"
        dis.service = rectest["service"].code
        dis.type_servi = "GLOBAL_BTE"
        dis.num_ipp = PatientGlobal.Ipp.value
        dis.num_sej = PatientGlobal.Sejour.value
        dis.id_servi = 2
        dis.mode = "S"
        dis.save()
        assert dis.pk > 0
        dis.delete_instance()

    def test_riedl_error_message_wo_liste(self, rectest):
        """Test create an output lot list"""
        with pytest.raises(MedianLibException) as e:
            err = FListeError()
            err.liste = "RIEDL LOT OUTPUT FAILED"
            err.mode = "S"
            err.message = "Test message"
            err.save()
        assert "Original list not found !" in str(e.value)

    def test_riedl_create_output_lot(self, rectest):
        """Test create an output lot list"""
        dis = Dispensation()
        dis.liste = "RIEDL LOT OUTPUT"
        dis.fusion = "SORTIE DE LOTS"
        dis.service = rectest["service"].code
        dis.type_servi = "RIEDL_BTE"
        dis.num_ipp = PatientGlobal.Ipp.value
        dis.num_sej = PatientGlobal.Sejour.value
        dis.id_servi = 2
        dis.mode = "S"
        dis.etat = "V"
        dis.save()
        assert dis.pk > 0

        err = FListeError()
        err.liste = dis.liste
        err.mode = dis.mode
        err.message = "Test message"
        err.save()

        dis.delete_instance()

        # Check if error message is clean
        with pytest.raises(DoesNotExist):
            FListeError.get(liste=dis.liste, mode=dis.mode)

    def test_riedl_create_output_lot_check_state_undone(self, rectest):
        """Test create an output lot list
           When we add and error message, the state change to I
        """
        dis = Dispensation()
        dis.liste = "RIEDL LOT OUTPUT"
        dis.fusion = "SORTIE DE LOTS"
        dis.service = rectest["service"].code
        dis.type_servi = "RIEDL_BTE"
        dis.num_ipp = PatientGlobal.Ipp.value
        dis.num_sej = PatientGlobal.Sejour.value
        dis.id_servi = 2
        dis.mode = "S"
        dis.etat = "V"
        dis.save()
        assert dis.pk > 0

        err = FListeError()
        err.liste = dis.liste
        err.mode = dis.mode
        err.message = "Test message"
        err.save()

        lect = Dispensation.get(liste=dis.liste, mode=dis.mode)
        assert lect.etat == "I"

        dis.delete_instance()

        # Check if error message is clean
        with pytest.raises(DoesNotExist):
            FListeError.get(liste=dis.liste, mode=dis.mode)

    def test_riedl_create_output_lot_check_state_done(self, rectest):
        """Test create an output lot list
           When we add and error message, the state change to I
           except for state done "S"
        """
        dis = Dispensation()
        dis.liste = "RIEDL LOT OUTPUT"
        dis.fusion = "SORTIE DE LOTS"
        dis.service = rectest["service"].code
        dis.type_servi = "RIEDL_BTE"
        dis.num_ipp = PatientGlobal.Ipp.value
        dis.num_sej = PatientGlobal.Sejour.value
        dis.id_servi = 2
        dis.mode = "S"
        dis.etat = "S"
        dis.save()
        assert dis.pk > 0

        err = FListeError()
        err.liste = dis.liste
        err.mode = dis.mode
        err.message = "Test message"
        err.save()

        lect = Dispensation.get(liste=dis.liste, mode=dis.mode)
        assert lect.etat == "S"

        dis.delete_instance()

        # Check if error message is clean
        with pytest.raises(DoesNotExist):
            FListeError.get(liste=dis.liste, mode=dis.mode)
