from peewee import (
    IntegerField,
    CharField,
    DateTimeField,
    BooleanField,
    AutoField,
    FloatField,
    TextField
)
from peewee import SQL, DoesNotExist, fn
from playhouse.hybrid import hybrid_method
from ..base import BaseModel, MedianLibException
from .stock import Stock
from .magasin import Zone, ZoneMag
from .dispensation import FItem, FListe
from ..constant import DRUGS_ANOMALY, DRUGS_FRACTION
import datetime

# !! use DRUGS_FRACTION constants instead !!
FRACTION = [
    25, 50,
    75, 100
]


class Product(BaseModel):
    pk = AutoField(column_name="x_pk", primary_key=True)
    reference = CharField(
        column_name='x_ref', unique=True, null=True, max_length=35,
        constraints=[SQL("DEFAULT ''")], default='')
    designation = CharField(
        column_name='x_desig', null=True, max_length=100,
        constraints=[SQL("DEFAULT ''")], default='')
    tmodif = DateTimeField(
        column_name="x_tmodif",
        null=True,
        constraints=[SQL("DEFAULT '0000-00-00 00:00:00'")],
    )
    tsortie = DateTimeField(
        column_name="x_tsortie",
        null=True,
        constraints=[SQL("DEFAULT '0000-00-00 00:00:00'")],
    )
    tentree = DateTimeField(
        column_name="x_tentree",
        null=True,
        constraints=[SQL("DEFAULT '0000-00-00 00:00:00'")],
    )
    cip = CharField(column_name="x_cip", null=True, constraints=[SQL("DEFAULT ''")], default="")
    ucd = CharField(column_name="x_ucd", null=True, constraints=[SQL("DEFAULT ''")], default="")
    dci = CharField(column_name="x_dci", constraints=[SQL("DEFAULT ''")], default="")
    forme = CharField(
        column_name="x_forme", max_length=20,
        constraints=[SQL("DEFAULT ''")], default=""
    )
    desig_bis = CharField(column_name="x_desig_bis", constraints=[SQL("DEFAULT ''")])
    # unit_gest = CharField(column_name="x_unit_gest", constraints=[SQL("DEFAULT ''")])
    # unit_admin = CharField(column_name="x_unit_admin", constraints=[SQL("DEFAULT ''")])
    coef_conv = FloatField(
        column_name="x_coef_conv", constraints=[SQL("DEFAULT 0.000")]
    )
    delai_peremp = IntegerField(
        column_name='x_delai_peremp', constraints=[SQL("DEFAULT 0")], default=0)
    externe = BooleanField(
        column_name='x_externe', null=True, constraints=[SQL("DEFAULT 0")], default=0)
    conversion_ucd = BooleanField(
        column_name='x_conversion_ucd', null=True, constraints=[SQL("DEFAULT 0")], default=0)
    multidose_par_uf = BooleanField(
        column_name='x_multidose_par_uf', constraints=[SQL("DEFAULT 0")], default=0)
    com_med = TextField(
        column_name='x_com_med', null=True, default="")
    bac = IntegerField(
        column_name='x_bac', constraints=[SQL("DEFAULT 0")], default=0)
    stup = BooleanField(
        column_name='x_stup', constraints=[SQL("DEFAULT 0")], default=0)
    risque = BooleanField(
        column_name='x_risque', constraints=[SQL("DEFAULT 0")], default=0)
    warn_pct = IntegerField(
        column_name='x_warn_pct', constraints=[SQL("DEFAULT 20")], default=20)
    tiroir_bas = IntegerField(
        column_name='x_tiroir_bas', null=True, constraints=[SQL("DEFAULT 0")], default=0)
    anomaly = CharField(
        column_name="x_anomalie", max_length=10,
        constraints=[SQL("DEFAULT ''")], default="")

    @hybrid_method
    def stock_liste(self, typemag, fraction):
        ret = 0
        try:
            stk = FItem().select(
                (fn.SUM(FItem.qte_dem) - fn.SUM(FItem.qte_serv)).alias("total")
            ).join(FListe, on=(FListe.liste == FItem.liste)).where(
                FListe.zone_fin == typemag, FListe.mode == 'E',
                FItem.reference == self.reference,
                FItem.fraction == fraction
            )
            if len(stk):
                ret = stk[0].total or 0
        except DoesNotExist:  # pragma: no cover
            pass
        return ret

    @hybrid_method
    def stock_magasin(self, magasin, fraction):
        """Retrieve the quantity in stock"""
        ret = 0
        try:
            stk = Stock().select(
                fn.SUM(Stock.quantite).alias("total")
            ).where(
                Stock.magasin == magasin, Stock.reference == self.reference,
                Stock.fraction == fraction,
                Stock.date_peremption > datetime.datetime.now()
            )
            if len(stk):
                ret = stk[0].total or 0
        except DoesNotExist:  # pragma: no cover
            ret = 0
        return ret

    @hybrid_method
    def stock_coupe(self, type_mag, fraction):
        """Retrieve the quantity in stock"""
        ret = 0
        try:
            stk = Stock().select(fn.SUM(Stock.quantite).alias("total")).join(
                ZoneMag, on=(Stock.magasin == ZoneMag.mag)
            ).join(
                Zone, on=(Zone.zone == ZoneMag.zone & Zone.appro == 1)
            ).where(
                Stock.zone_admin == type_mag,
                Stock.reference == self.reference,
                Stock.fraction == fraction
            )
            if len(stk):
                ret = stk[0].total or 0
        except DoesNotExist:  # pragma: no cover
            ret = 0
        return ret

    class Meta:
        table_name = "f_ref"
        indexes = (
            (('x_ref',), True),
        )

    def save(self, force_insert=False, only=None):
        """Enforce control on each information passed"""
        if self.anomaly and self.anomaly not in [da[0] for da in DRUGS_ANOMALY]:
            raise ValueError('Anomaly value is incorrect')
        return super().save(force_insert, only)


class Seuil(BaseModel):
    pk = AutoField(
        column_name='x_pk', primary_key=True)
    reference = CharField(
        column_name='x_ref', constraints=[SQL("DEFAULT ''")])
    zone = CharField(
        column_name='x_zone', constraints=[SQL("DEFAULT ''")])
    stock_mini = FloatField(
        column_name='x_stk_mini', constraints=[SQL("DEFAULT 0.000")])
    stock_maxi = FloatField(
        column_name='x_stk_maxi', constraints=[SQL("DEFAULT 0.000")])
    fraction = IntegerField(
        column_name='x_fraction', constraints=[SQL("DEFAULT 100")])

    class Meta:
        table_name = 'f_seuil'
        indexes = (
            (('x_ref', 'x_zone'), True),
        )

    def save(self, force_insert=False, only=None):
        """Enforce control on each information passed"""
        if self.fraction not in DRUGS_FRACTION:
            raise ValueError('Fraction value incorrect')
        return super().save(force_insert, only)


class ContainerFormat(BaseModel):
    """Blacklist batch lot from the supplier autorisation code"""
    pk = AutoField(
        column_name='x_pk', primary_key=True)
    code = CharField(
        column_name='x_format', max_length=10,
        constraints=[SQL("DEFAULT ''")], default="")
    dimension_x = IntegerField(
        column_name='x_dim_x', constraints=[SQL("DEFAULT 0")], default=0)
    dimension_y = IntegerField(
        column_name='x_dim_y', constraints=[SQL("DEFAULT 0")], default=0)
    dimension_z = IntegerField(
        column_name='x_dim_z', constraints=[SQL("DEFAULT 0")], default=0)
    orientation = CharField(
        column_name='x_orient', max_length=1,
        constraints=[SQL("DEFAULT ''")], default="Y")
    divide_number = IntegerField(
        column_name='x_nb_div', constraints=[SQL("DEFAULT 0")], default=0)
    color = IntegerField(
        column_name='x_couleur', constraints=[SQL("DEFAULT 0")], default=12632256)
    measurement_x_mm = IntegerField(
        column_name='x_mm_x', constraints=[SQL("DEFAULT 0")], default=0)
    measurement_y_mm = IntegerField(
        column_name='x_mm_y', constraints=[SQL("DEFAULT 0")], default=0)
    measurement_z_mm = IntegerField(
        column_name='x_mm_z', constraints=[SQL("DEFAULT 0")], default=0)
    type_tray = CharField(
        column_name='x_type_bac', max_length=10,
        constraints=[SQL("DEFAULT ''")], default="")

    @hybrid_method
    def volume(self,):
        """Compute the volume for this """
        return (self.measurement_x_mm * self.measurement_y_mm * self.measurement_z_mm) or 0

    def save(self, force_insert=False, only=None):
        """Enforce control on each information passed"""
        # We retrieve the liste
        if self.dimension_x < 0:
            raise MedianLibException("Width must greather or equal to 0")
        if self.dimension_y < 0:
            raise MedianLibException("Height must greather or equal to 0")
        if self.dimension_z < 0:
            raise MedianLibException("Depth must greather or equal to 0")
        return super().save(force_insert, only)

    class Meta:
        table_name = 'f_format'
        indexes = (
            (('x_format',), True),
        )


class LotRetire(BaseModel):
    """Blacklist batch lot from the supplier autorisation code"""
    pk = AutoField(
        column_name='x_pk', primary_key=True)
    ucd = CharField(
        column_name='x_ucd', max_length=20,
        constraints=[SQL("DEFAULT ''")], default="")
    lot = CharField(
        column_name='x_lot', max_length=20,
        constraints=[SQL("DEFAULT ''")], default="")

    class Meta:
        table_name = 'f_lot_retire'
        indexes = (
            (('x_ucd', 'x_lot'), True),
            (('x_ucd',), False),
        )


class TopFive(BaseModel):
    pk = AutoField(
        column_name='x_pk', primary_key=True)
    cle = CharField(
        column_name='x_cle', max_length=10,
        constraints=[SQL("DEFAULT ''")])
    val = CharField(
        column_name='x_val', max_length=20,
        constraints=[SQL("DEFAULT ''")])
    poste = CharField(
        column_name='x_poste', max_length=20,
        constraints=[SQL("DEFAULT ''")])
    user = CharField(
        column_name='x_user', max_length=20,
        constraints=[SQL("DEFAULT ''")])
    mode = BooleanField(
        column_name='x_mode', constraints=[SQL("DEFAULT 0")])
    nb = IntegerField(
        column_name='x_nb', null=True, constraints=[SQL("DEFAULT 0")])

    class Meta:
        table_name = 'f_top5'
        indexes = (
            (('x_cle', 'x_val', 'x_poste', 'x_mode'), True),
        )
