# flake8: noqa
import pytest
from peewee import DoesNotExist, IntegrityError
from median.models import Product, Ucd, Cip, GtinAlias, Seuil, ContainerFormat


class TestModelProduct:
    def test_unknown_product(self):
        """Test d'un produit inexistant"""
        with pytest.raises(DoesNotExist):
            Product.get(reference="INCONNU")

    def test_select_product(self):
        """Verification du produit deverminage"""
        pr = Product.get(reference="DEVERMINAGE")
        assert pr.pk > 0

    def test_create_product(self):
        """Creation d'un produit de test"""
        pr = Product()
        pr.reference = "ECO1234"
        pr.designation = "Eco Medoc"
        pr.save()
        assert pr.pk > 0

    def test_modify_product(self):
        """Modify product (eg: astus only)"""
        pr = Product.get(reference="ECO1234")
        assert pr.externe == False
        pr.externe = True
        pr.save()
        assert pr.externe == True

    def test_delete_product(self):
        """Suppression du produitd e test"""
        pr = Product.get(reference="ECO1234")
        pr.delete_instance()

    def test_create_product_with_known_anomaly(self):
        """Creation d'un produit de test avec anomalie connue"""
        pr = Product()
        pr.reference = "ECO1234"
        pr.designation = "Eco Medoc"
        pr.anomaly = "UCD_DBL"
        pr.save()
        assert pr.pk > 0
        pr.delete_instance()

    def test_create_product_with_unknown_anomaly(self):
        """Creation d'un produit de test avec anomalie connue"""
        with pytest.raises(ValueError):
            pr = Product()
            pr.reference = "ECO1234"
            pr.designation = "Eco Medoc"
            pr.anomaly = "UNKNOWN"
            pr.save()

    def test_create_product_ucd(self):
        """Creation d'un produit de test avec UCD"""
        pr = Product()
        pr.reference = "ECO9999"
        pr.designation = "Eco Medoc"
        pr.save()
        assert pr.pk > 0
        ucd = Ucd()
        ucd.reference = pr.reference
        ucd.ucd = "9047693"
        ucd.save()
        assert ucd.pk > 0
        ucd.delete_instance()
        pr.delete_instance()

    def test_create_product_ucd_cip(self):
        """Creation d'un produit de test avec UCD et CIP"""
        pr = Product()
        pr.reference = "ECO9999"
        pr.designation = "Eco Medoc"
        pr.save()
        assert pr.pk > 0
        ucd = Ucd()
        ucd.reference = pr.reference
        ucd.ucd = "9047693"
        ucd.save()
        assert ucd.pk > 0
        cip = Cip()
        cip.ucd = ucd.ucd
        cip.cip = "3400937140423"
        cip.save()
        assert cip.pk > 0
        assert cip.dossier == "0"
        cip.delete_instance()
        ucd.delete_instance()
        pr.delete_instance()

    def test_gtin_alias_creation(self):
        pr = Product()
        pr.reference = "ECO9999"
        pr.designation = "Eco Medoc"
        pr.save()
        assert pr.pk > 0
        ucd = Ucd()
        ucd.reference = pr.reference
        ucd.ucd = "9047693"
        ucd.save()
        assert ucd.pk > 0
        cip = Cip()
        cip.ucd = ucd.ucd
        cip.cip = "3400937140423"
        cip.save()
        assert cip.pk > 0
        assert cip.dossier == "0"
        alias = GtinAlias()
        alias.source = "25500100000012"
        alias.destination = "3400937140423"
        alias.save()
        assert alias.pk > 0
        alias.delete_instance()
        cip.delete_instance()
        ucd.delete_instance()
        pr.delete_instance()


    def test_create_seuil_fraction_100_ok(self):
        """Creation d'un seuil pour la fraction 100"""
        s = Seuil()
        s.reference = "4556485"
        s.zone = "CUEILLETTE_100"
        s.stock_mini = 14
        s.stock_maxi = 28
        s.fraction = 100
        s.save()
        assert s.pk > 0
        s.delete_instance()

    def test_create_seuil_fraction_50_ok(self):
        """Creation d'un seuil pour la fraction 30"""
        s = Seuil()
        s.reference = "4556485"
        s.zone = "CUEILLETTE_100"
        s.stock_mini = 14
        s.stock_maxi = 28
        s.fraction = 50
        s.save()
        assert s.pk > 0
        s.delete_instance()

    def test_create_seuil_fraction_30_fail(self):
        """interdire la creation d'un seuil pour la fraction 30"""
        with pytest.raises(ValueError):
            s = Seuil()
            s.reference = "4556485"
            s.zone = "CUEILLETTE_100"
            s.stock_mini = 15
            s.stock_maxi = 30
            s.fraction = 30
            s.save()

    def test_product_stock_unknown_mag(self):
        """Test produit sur magasin existant"""
        pr = Product()
        pr.reference = "ECO1234"
        pr.designation = "Eco Medoc"
        pr.save()
        assert pr.pk > 0

        ret = pr.stock_magasin("UNKNOWN", 100)
        assert ret == 0

        pr.delete_instance()

    def test_container_format_creation(self):
        """Create a container format"""
        cf = ContainerFormat()
        cf.code = "BACTEST"
        cf.save()

        cf.delete_instance()

    def test_container_format_creation_twice(self):
        """Create a container format twice"""
        cf = ContainerFormat()
        cf.code = "BACTEST"
        cf.save()

        with pytest.raises(IntegrityError):
            cf2 = ContainerFormat()
            cf2.code = "BACTEST"
            cf2.save()

        cf.delete_instance()
