from peewee import DoesNotExist, OperationalError
from .utils import logger
from .models import (
    PrescriptionItem, MagasinService, LogAstus, TopFive,
    HistoriqueMultidoses, MessageAstus, FListeError,
    Patient, Sejour, Service, WebviewTaquin, WebviewDemiTaquin,
    WebviewEmplacement, WebviewPatientHisto, LogInfo, LotRetire,
    GtinAlias
)
from .constant import PatientGlobal


class DbMigrator:

    def __init__(self, db_instance):
        """Initilise the migraotr class with the database selected"""
        self.db = db_instance
        logger.debug('Initilize migrator instance')

    def migrate(self, force=None):
        """Launch the migration script
            force argument, drop  some viewe
        """
        if force:
            logger.info('Migrate force activate')
            self.drop_sql_view('webview_patient_histo', WebviewPatientHisto)
            self.drop_sql_view('webview_emplacement', WebviewEmplacement)
            self.drop_sql_view('webview_demi_taquin', WebviewDemiTaquin)
            self.drop_sql_view('webview_taquin', WebviewTaquin)

        self.check_model('f_prescription_item', PrescriptionItem)
        self.check_model('f_mag_dest', MagasinService)
        self.check_model('f_log_astus', LogAstus)
        self.check_model('f_top5', TopFive)
        self.check_model('f_lot_retire', LotRetire)
        self.check_model('f_histo_multidoses', HistoriqueMultidoses)
        self.check_model('f_message_astus', MessageAstus)
        self.check_model('f_liste_error', FListeError)
        self.check_model('f_log_info', LogInfo)
        self.check_model('f_cip_alias', GtinAlias)
        self.check_field('f_user', 'x_astup', "ALTER TABLE f_user ADD COLUMN x_astup tinyint(1) DEFAULT '0';")
        self.check_field('f_user', 'x_inventaire', "ALTER TABLE f_user ADD COLUMN x_inventaire tinyint(1) DEFAULT '0';")
        self.check_field('f_user', 'x_login', "ALTER TABLE f_user ADD COLUMN x_login VARCHAR(30) DEFAULT '';")
        self.check_field('f_user', 'x_url', "ALTER TABLE f_user ADD COLUMN x_url VARCHAR(250) DEFAULT '';")
        self.check_field('f_user', 'x_badge', "ALTER TABLE f_user ADD COLUMN x_badge varchar(16) DEFAULT '';")
        self.check_field('f_user', 'x_email', "ALTER TABLE f_user ADD COLUMN x_email VARCHAR(320) DEFAULT '';")
        self.check_field('f_user', 'x_lang', "ALTER TABLE f_user ADD COLUMN x_lang VARCHAR(10) DEFAULT '';")
        self.check_field('f_user', 'x_import_flag',
                         "ALTER TABLE f_user ADD COLUMN x_import_flag TINYINT(1) NULL DEFAULT 0;")
        self.check_field('f_user', 'x_maintenance',
                         "ALTER TABLE f_user ADD COLUMN x_maintenance TINYINT(1) NULL DEFAULT 0;")
        self.check_field('f_user', 'x_retry',
                         "ALTER TABLE f_user ADD COLUMN x_retry INT(11) NULL DEFAULT 5;")
        self.check_field('f_user', 'x_passwd_web',
                         "ALTER TABLE f_user ADD COLUMN x_passwd_web VARCHAR(320) DEFAULT '';")
        self.check_field('f_user', 'x_is_enabled',
                         "ALTER TABLE f_user ADD COLUMN x_is_enabled TINYINT(1) NULL DEFAULT 1;")
        self.check_field('f_user', 'x_is_temporary',
                         "ALTER TABLE f_user ADD COLUMN x_is_temporary TINYINT(1) NULL DEFAULT 1;")

        self.check_field('f_profil', 'x_visu', "ALTER TABLE f_profil ADD COLUMN x_visu TINYINT(1) NULL DEFAULT 0;")
        self.check_field('f_profil', 'x_edit', "ALTER TABLE f_profil ADD COLUMN x_edit TINYINT(1) NULL DEFAULT 0;")
        self.check_field('f_ref', 'x_com_med', "ALTER TABLE f_ref ADD COLUMN x_com_med VARCHAR(100) DEFAULT '';")
        self.check_field('f_ref', 'x_bac', "ALTER TABLE f_ref ADD COLUMN x_bac INT(11) NULL DEFAULT 0;")
        self.check_field('f_ref', 'x_stup', "ALTER TABLE f_ref ADD x_stup TINYINT(1) NOT NULL DEFAULT 0;")
        self.check_field('f_ref', 'x_risque', "ALTER TABLE f_ref ADD x_risque TINYINT(1) NOT NULL DEFAULT 0;")
        self.check_field('f_ref', 'x_warn_pct', "ALTER TABLE f_ref ADD x_warn_pct INT NOT NULL DEFAULT 20;")
        self.check_field('f_ref', 'x_anomalie', "ALTER TABLE f_ref ADD COLUMN x_anomalie VARCHAR(10) DEFAULT '';")
        self.check_field('f_reform', 'x_ordre', "ALTER TABLE f_reform ADD COLUMN x_ordre INT(11) NULL DEFAULT 0;")
        self.check_field('f_ucd_cip', 'x_chrono',
                         "ALTER TABLE f_ucd_cip ADD COLUMN x_chrono DATETIME DEFAULT '0000-00-00 00:00:00';")
        self.check_field('f_ucd_cip', 'x_update',
                         "ALTER TABLE f_ucd_cip ADD COLUMN x_update DATETIME DEFAULT '0000-00-00 00:00:00';")
        self.check_field('f_ucd_cip', 'x_last_user',
                         "ALTER TABLE f_ucd_cip ADD COLUMN x_last_user VARCHAR(35) DEFAULT '';")
        self.check_field('f_mag', 'x_last_reap',
                         "ALTER TABLE f_mag ADD COLUMN x_last_reap DATETIME DEFAULT '0000-00-00 00:00:00';")
        self.check_field('f_mag', 'x_type_machine',
                         "ALTER TABLE f_mag ADD COLUMN x_type_machine VARCHAR(20) DEFAULT '';")
        self.check_field('f_stock', 'x_qte_blister_bac',
                         "ALTER TABLE f_stock ADD COLUMN x_qte_blister_bac INT(11) NULL DEFAULT 0;")
        self.check_field('f_stock', 'x_nb_dose_cut',
                         "ALTER TABLE f_stock ADD COLUMN x_nb_dose_cut INT(11) NULL DEFAULT 0;")
        self.check_field('f_stock', 'x_version', "ALTER TABLE f_stock ADD COLUMN x_version INT(11) NULL DEFAULT 0;")
        self.check_field('f_stock', 'x_DU', "ALTER TABLE f_stock ADD COLUMN x_DU INT(11) NULL DEFAULT 0;")
        self.check_field('f_stock', 'x_qte_prod_blister',
                         "ALTER TABLE f_stock ADD COLUMN x_qte_prod_blister INT(11) NULL DEFAULT 0;")
        self.check_field('f_ref', 'x_tiroir_bas',
                         "ALTER TABLE f_ref ADD COLUMN x_tiroir_bas TINYINT(1) NULL DEFAULT 0;")
        self.check_field('f_adr', 'x_qte_posage',
                         "ALTER TABLE f_adr ADD COLUMN x_qte_posage INT(11) NULL DEFAULT 0;")
        self.check_field('f_adr', 'x_nouv_qte_posage',
                         "ALTER TABLE f_adr ADD COLUMN x_nouv_qte_posage INT(11) NULL DEFAULT 0;")
        self.check_field('f_adr', 'x_emplacement',
                         "ALTER TABLE f_adr ADD COLUMN x_emplacement VARCHAR(20) NULL DEFAULT '';")
        self.check_field('f_adr', 'x_nouv_emplacement',
                         "ALTER TABLE f_adr ADD COLUMN x_nouv_emplacement varchar(20) NULL DEFAULT '';")
        self.check_field('f_adr', 'x_stup',
                         "ALTER TABLE f_adr ADD COLUMN x_stup TINYINT(1) NULL DEFAULT 0;")

        self.check_field('f_adr', 'x_bloque_reappro',
                         "ALTER TABLE f_adr ADD COLUMN x_bloque_reappro TINYINT(1) NOT NULL DEFAULT 0;")
        self.check_field('f_adr', 'x_service',
                         "ALTER TABLE f_adr ADD COLUMN x_service TINYINT(1) NULL DEFAULT 0;")
        self.check_field('f_adr', 'x_bloque_msg',
                         "ALTER TABLE f_adr ADD COLUMN x_bloque_msg varchar(100) NULL DEFAULT '';")

        # Table f_dest for Carnet and Riedl Exit
        self.check_field('f_dest', 'x_type_carnet',
                         "ALTER TABLE f_dest ADD COLUMN x_type_carnet INT(11) NULL DEFAULT '1'")
        self.check_field('f_dest', 'x_secteur', "ALTER TABLE f_dest ADD COLUMN x_secteur VARCHAR(40) NULL DEFAULT ''")
        self.check_field('f_dest', 'x_adr1', "ALTER TABLE f_dest ADD COLUMN x_adr1 VARCHAR(40) NULL DEFAULT ''")
        self.check_field('f_dest', 'x_adr2', "ALTER TABLE f_dest ADD COLUMN x_adr2 VARCHAR(40) NULL DEFAULT ''")
        self.check_field('f_dest', 'x_adr3', "ALTER TABLE f_dest ADD COLUMN x_adr3 VARCHAR(40) NULL DEFAULT ''")
        self.check_field('f_dest', 'x_adr4', "ALTER TABLE f_dest ADD COLUMN x_adr4 VARCHAR(40) NULL DEFAULT ''")
        self.check_field('f_dest', 'x_tk_mode',
                         "ALTER TABLE f_dest ADD COLUMN x_tk_mode VARCHAR(2) NOT NULL DEFAULT 'NO'")
        self.check_field('f_mag', 'x_split_liste',
                         "ALTER TABLE f_mag ADD COLUMN x_split_liste TINYINT(1) NULL DEFAULT 0;")
        self.check_field('f_mag', 'x_libelle',
                         "ALTER TABLE f_mag ADD COLUMN x_libelle varchar(35) NOT NULL DEFAULT '';")
        self.check_field('f_poste', 'x_proc_pid', "ALTER TABLE f_poste ADD COLUMN x_proc_pid INT(11) NULL DEFAULT 0;")
        self.check_field('f_stock', 'x_capa', "ALTER TABLE f_stock ADD COLUMN x_capa INT(11) NULL DEFAULT 0;")
        self.check_field('f_stock', 'x_cycleVie', "ALTER TABLE f_stock ADD COLUMN x_cycleVie INT(11) DEFAULT 0;")
        self.check_field('f_stock', 'x_indexprise',
                         "ALTER TABLE f_stock ADD COLUMN x_indexprise SMALLINT(6) NULL DEFAULT 0;")
        self.check_field('f_stock', 'x_cip',
                         "ALTER TABLE f_stock ADD COLUMN x_cip varchar(13) NOT NULL DEFAULT '';")
        self.check_field('f_item', 'x_liste_orig',
                         "ALTER TABLE f_item ADD COLUMN x_liste_orig VARCHAR(60) NULL DEFAULT ''")
        self.check_field('f_item', 'x_posologie', "ALTER TABLE f_item ADD COLUMN x_posologie TEXT NULL DEFAULT ''")
        self.check_field('f_liste', 'x_sequence',
                         "ALTER TABLE f_liste ADD COLUMN x_sequence VARCHAR(25) NULL DEFAULT ''")
        self.check_field('f_liste', 'x_interface',
                         "ALTER TABLE f_liste ADD COLUMN x_interface VARCHAR(15) NULL DEFAULT ''")
        self.check_field('f_liste', 'x_liste_orig',
                         "ALTER TABLE f_liste ADD COLUMN x_liste_orig VARCHAR(60) NULL DEFAULT ''")
        self.check_field('f_liste', 'x_num_face',
                         "ALTER TABLE f_liste ADD COLUMN x_num_face INT(11) NULL DEFAULT 0;")
        self.check_field('f_liste_error', 'x_message',
                         "ALTER TABLE f_liste_error ADD COLUMN x_message TEXT NULL DEFAULT ''")
        # If pk not exists, we drop the table to recreate it
        self.check_field('f_liste_error', 'x_pk',
                         "DROP TABLE IF EXISTS f_liste_error;")
        self.check_model('f_liste_error', FListeError)

        self.check_model_index(LotRetire)

        self.add_service('DEENOVA', 'Service de test')
        self.add_global_ipp_episode()
        # TODO: Drop all view before restore (force mode ?)
        logger.info('Check views')
        self.check_sql_view('webview_taquin', WebviewTaquin)
        self.check_sql_view('webview_demi_taquin', WebviewDemiTaquin)
        self.check_sql_view('webview_emplacement', WebviewEmplacement)
        self.check_sql_view('webview_patient_histo', WebviewPatientHisto)
        logger.info('End check views')

    def check_model(self, table_name, model_class):
        """Check if table exists, and create it if missing"""
        if not self.db.table_exists(table_name):
            logger.warning('Table %s does not exists, we create it from %s' % (table_name, model_class))
            self.db.create_tables([model_class])
        else:
            logger.info('Table %s exists' % (table_name,))

        return True

    def check_model_index(self, model_class):
        """Check if index exists, creat it if missing"""
        unique_template = """ALTER TABLE %s ADD CONSTRAINT %s UNIQUE KEY (%s);"""
        index_template = """CREATE INDEX %s USING BTREE ON %s (%s);"""
        table_name = model_class._meta.table_name
        # print(table_name)
        for idx in model_class._meta.fields_to_index():
            exp = idx._expressions
            fld_list = [x.column.name for x in exp]

            if idx._unique:
                ddl = unique_template % (table_name, idx._name, ", ".join(fld_list))
            else:
                ddl = index_template % (idx._name, table_name, ", ".join(fld_list))

            try:
                self.db.execute_sql(ddl)
            except OperationalError:
                logger.warning("index %s exists, do nothing" % idx._name)

    def check_sql_view(self, view_name, model_class):
        """Check if view exists, if not create it"""
        vws = [v.name for v in self.db.get_views()]
        if view_name not in vws:
            logger.info(' -> View %s is missing, we create it' % view_name)
            kls = model_class()
            kls.create_view()

    def drop_sql_view(self, view_name, model_class):
        """Check if view exists, if exists drop it"""
        vws = [v.name for v in self.db.get_views()]
        if view_name in vws:
            logger.info(' -> View %s exists, we delete it' % view_name)
            kls = model_class()
            kls.delete_view(True)

    def check_field(self, table_name, field_name, ddl_str=None):
        """check if the field exists"""
        field_found = False
        for col in self.db.get_columns(table_name):
            if col.name == field_name:
                field_found = True
                break

        if field_found:
            logger.info('Column %s exists in table %s' % (field_name, table_name))
        else:
            logger.warning('Column %s does not exists in table %s' % (field_name, table_name))
            if ddl_str is None:
                return False
            self.db.execute_sql(ddl_str)

        return True

    def delete_model(self, table_name, model_class):
        """Delete the table relate to the model"""
        if self.db.table_exists(table_name):
            self.db.drop_tables([model_class])
        else:
            logger.warning('Table %s not found, cannot delete it' % (table_name,))

    def delete_field(self, table_name, field_name):
        """Delete a field on a database"""
        query = "ALTER TABLE %s DROP COLUMN %s;" % (table_name, field_name)
        self.db.execute_sql(query)
        return True

    def add_service(self, wardcode, wardname):
        """Check if DEENOVA appear on f_dest"""
        try:
            Service.get(code=wardcode)
        except DoesNotExist:
            ser = Service()
            ser.code = wardcode
            ser.libelle = wardname
            ser.save()
        return True

    def add_global_ipp_episode(self):
        """
        If Global IPP and Episode not exists, we create it
        """
        try:
            Patient.get(ipp='GLOBAL')
        except DoesNotExist:
            logger.info('Global patient not exists, we create it')
            p = Patient()
            p.ipp = PatientGlobal.Ipp.value
            p.prenom = 'GL'
            p.nom = 'GL'
            p.sexe = 'H'
            p.save()

        try:
            Sejour.get(ipp='GLOBAL', sejour='GLOBAL')
        except DoesNotExist:
            logger.info('Global episode not exists, we create it')
            s = Sejour()
            s.ipp = PatientGlobal.Ipp.value
            s.sejour = PatientGlobal.Sejour.value
            s.uf_alit = '-'
            s.uf_hosp = '-'
            s.save()
