import sys
import os
import logging
import time

from flask import (
    Flask, session, redirect, request,
    send_from_directory, send_file, jsonify
)
from flask.json import JSONEncoder
from datetime import date
from pathlib import Path
from flask_restful import Api
from common.database import sync_profil
from common.status import HTTP_404_NOT_FOUND
from ressources.datatables import add_dt_route
from ressources.ui import add_ui_route
from ressources import add_res_route
from median.auth import user_auth
from median.models import User


class MedianJSONEncoder(JSONEncoder):
    def default(self, o):
        if isinstance(o, date):
            return o.isoformat()

        return super().default(o)


# Les fichiers HTML sont fournis en statique.
html_dir = os.environ.get('MEDIAN_HTML_DIR', os.path.join('.', 'html'))
static_dir = os.environ.get('MEDIAN_STATIC_DIR', os.path.join('.', 'static'))

# Le fichier de log se configure dans IIS
log_file = os.environ.get('MEDIAN_LOG_FILE', '%s-median_leger.log' % time.strftime('%Y-%m-%d'))
logging.basicConfig(
    handlers=[logging.FileHandler(log_file, encoding='utf-8')],
    level=logging.DEBUG,
    format='%(asctime)s  %(name)-20s  %(levelname)-6s: %(message)s'
)
logger = logging.getLogger('median.webserver')

# Key use to crypt session cookie
if os.environ.get('MEDIAN_SECRET_KEY'):
    s_key = bytes(os.environ['MEDIAN_SECRET_KEY'], encoding='utf8')
else:
    s_key = logger.debug(os.urandom(16))

app = Flask(__name__)
app.secret_key = s_key
app.json_encoder = MedianJSONEncoder

api = Api(app, prefix='/api')

add_dt_route(api)
add_res_route(api)
add_ui_route(api)


@app.before_request
def before_request():
    """Use for reverse proxy with https enabled"""
    scheme = request.headers.get('X-Forwarded-Proto')
    if scheme and scheme == 'http' and request.url.startswith('http://'):
        url = request.url.replace('http://', 'https://', 1)
        code = 301
        return redirect(url, code=code)


@app.route('/')
def home():
    """Home page"""
    if not session.get('username', False):
        return redirect('/login')
    return redirect('/home')


@app.route('/home')
def homepage():
    """Launch the homepage,
    TODO: Can be defined per user
    """
    if not session.get('username', False):
        return redirect('/login')
    return redirect('/reference.html')


@app.route('/login', methods=['GET', 'POST'])
def login():
    """Send user to login form"""
    if request.method == 'POST':
        # vérifier username & mdp median.auth.check_auth
        user_id = user_auth(request.form['username'], request.form['password'])

        if user_id:
            usr = User.get(pk=user_id)
            session['username'] = usr.username
            session['email'] = usr.email
            session['user_id'] = user_id
            session['user_lang'] = os.environ.get('MEDIAN_USER_LANG', 'fr_FR')
            return redirect('/reference.html')
        else:
            logger.warning('Tentative de connexion échouée')
            return redirect('/login2.html')

    return redirect('/login.html')


@app.route('/logout')
def logout():
    """On deconnecte l'utilisateur et nettoie sa session"""
    session.pop('username', None)
    session.pop('user_id', None)
    logger.info('Utilisateur déconnecté')
    return redirect('/login.html')


@app.route('/static/<path:chemin>', methods=['GET'])
def fichier_static(chemin):
    """On recupere et renvoi les fichiers statiques tel quel"""
    return send_from_directory(static_dir, chemin)


@app.route('/<path:chemin>', methods=['GET'])
def fichier_html(chemin):
    """On recupere et renvoi les page HTML telle quelle"""
    return send_from_directory(html_dir, chemin)


@app.route('/status', methods=['GET'])
def app_status():
    """retourne les informations utiles au debuggage"""
    if not session.get('username', False):
        return redirect('/login')
    dbinfo = {
        'name': os.environ.get('MEDIAN_DB_NAME', 'unknown'),
        'server': os.environ.get('MEDIAN_DB_HOST', 'unknown'),
        'user': os.environ.get('MEDIAN_DB_USER', 'unknown'),
        'port': os.environ.get('MEDIAN_DB_PORT', 'unknown'),
    }
    pyinfo = {
        'version': sys.version,
        'major': sys.version_info.major,
        'minor': sys.version_info.minor,
        'micro': sys.version_info.micro,
        'release': sys.version_info.releaselevel,
        'path': os.environ.get('PYTHONPATH', 'N/A'),
        'wsgi': os.environ.get('WSGI_HANDLER', 'N/A'),
    }
    return jsonify(database=dbinfo, python=pyinfo)


@app.route('/download/<string:filename>')
def download_file(filename):
    cr_folder = os.environ.get('REPORTING_FOLDER', '.')
    fp = Path(cr_folder).joinpath(filename)
    if not fp.exists():
        logger.error('Download: file %s not found!' % filename)
        return {'message': 'File not found', 'filename': filename}, HTTP_404_NOT_FOUND
    return send_file(fp, as_attachment=True)


@app.cli.command("migrate")
def migrate_app():
    """Update database catalog"""
    from common.database import check_model, sync_menu, median_upgrade, median_information
    from common.models import WebLang, WebMenu, WebMenuTranslation, WebForm
    dbi = median_information()
    print('Database information\nHost: %(server)s\nPort: %(port)s\nName: %(name)s\nUser: %(user)s\n-----------' % dbi)
    check_model('web_lang', WebLang, 'web_lang.json')
    check_model('web_menu', WebMenu, 'web_menu.json')
    sync_menu('web_menu', WebMenu, 'web_menu.json')
    check_model('web_menu_i18n', WebMenuTranslation)
    check_model('web_form_i18n', WebForm)
    median_upgrade()
    sync_profil()
    print('Migration done')


@app.cli.command("init")
def init_data():
    """Add initiate data"""
    from common.database import median_information
    from common.migration import add_default_libelle, add_default_rights, add_default_interfaces
    print("Initilize datas")
    dbi = median_information()
    print('Database information\nHost: %(server)s\nPort: %(port)s\nName: %(name)s\nUser: %(user)s\n-----------' % dbi)
    add_default_libelle()
    add_default_rights()
    add_default_interfaces()
    print("End initialize data")


@app.cli.command("fix")
def fix_datas():
    from common.fix import fix_foreign_key, fix_datas
    fix_foreign_key()
    fix_datas()


@app.cli.command("i18n")
def i18n_app():
    """read and insert catalog fro each lang"""
    import glob
    from common.translate import locales_dir, form_i18n_file_import, menu_i18n_file_import
    print("Synchronise lang in the database")
    print("Locales folder: " + locales_dir)
    for fold in glob.glob(locales_dir + '/**/LC_MESSAGES'):
        print("Update " + fold.split(os.sep)[-2] + " translation")
        form_i18n_file_import(os.path.join(fold, 'form.po'), fold.split(os.sep)[-2])
        menu_i18n_file_import(os.path.join(fold, 'menu.po'), fold.split(os.sep)[-2])
    print("Sync lang done.")


@app.cli.command("pot2po")
def pot_to_po():
    """Update PO files with POT template"""
    from common.translate import availables_locales, locales_dir, convert_pot2po

    potfile = os.path.join(locales_dir, 'form.pot')
    for al in availables_locales:
        pofile = os.path.join(locales_dir, al, 'LC_MESSAGES', 'form.po')
        print("Update form %s po file" % al)
        convert_pot2po(pofile, potfile)

    potfile = os.path.join(locales_dir, 'menu.pot')
    for al in availables_locales:
        pofile = os.path.join(locales_dir, al, 'LC_MESSAGES', 'menu.po')
        print("Update menu %s po file" % al)
        convert_pot2po(pofile, potfile)

    print('Merge PO with catalogs done')


@app.cli.command("riedl")
def riedl_management():
    from common.database import median_information
    from common.migration import riedl_migration
    dbi = median_information()
    print('Database information\nHost: %(server)s\nPort: %(port)s\nName: %(name)s\nUser: %(user)s\n-----------' % dbi)
    riedl_migration()
    print("End of Riedl migration")
