from flask_restful import Resource, reqparse
# from flask import session
from median.models import Ucd, Cip
from peewee import DoesNotExist, fn
import logging

logger = logging.getLogger('median.webserver')

parser = reqparse.RequestParser()
parser.add_argument('product_ref')
parser.add_argument('select')
parser.add_argument('draw', type=int, help='Draw parameter for Datatables')
parser.add_argument('start', type=int, help='Start parameter for Datatables')
parser.add_argument('length', type=int, help='Limit parameter for Datatables')
parser.add_argument('order[0][column]', type=int, help='Ord parameter for DT')
parser.add_argument('order[0][dir]', help='Ord parameter for Datatables')
parser.add_argument('ucd')
parser.add_argument('cip')
parser.add_argument('qte_pass')
parser.add_argument('qte_boite')
parser.add_argument('qte_blister')
parser.add_argument('index')


class DataTables(Resource):
    def render(self, datas, draw, recordsTotal, recordsFiltered, error=None):
        o = {
            'draw': draw,
            'recordsTotal': recordsTotal,
            'recordsFiltered': recordsFiltered,
            'data': datas
        }
        if error is not None:
            o['error'] = error
        return o


class ApiUcdCip(DataTables):

    def get(self, ref):
        args = parser.parse_args()
        v_draw = args['draw']

        try:
            total_ucd_cip_count = (
                Ucd.select()
                .join(Cip, on=(Ucd.ucd == Cip.ucd))
                .where(Ucd.reference == ref)
                .count())
            filtered_ucd_cip = (
                Cip.select(Cip.ucd, Cip.cip, Cip.qt_pass, Cip.qt_boite, Cip.qt_blister, Cip.dossier)
                .join(Ucd, on=(Ucd.ucd == Cip.ucd))
                .where(Ucd.reference == ref)).order_by(Cip.ucd, Cip.cip, Cip.dossier)

            logger.debug('Lines : %s.' % len(filtered_ucd_cip))

            return self.render([{
                'ucd': s.ucd,
                'cip': s.cip,
                'qte_pass': s.qt_pass,
                'qte_boite': s.qt_boite,
                'qte_blister': s.qt_blister,
                'index': s.dossier
            } for s in filtered_ucd_cip],
                v_draw,
                total_ucd_cip_count,
                filtered_ucd_cip.count())

        except DoesNotExist:
            logger.error('Get Ucd_Cip raised a DoesNotExist exception')
            return self.render([], v_draw, 0, 0)
        except Exception as error:
            logger.error('Get Ucd_Cip raised an exception: ', error.args)
            return self.render([], v_draw, 0, 0, error.args)

    def post(self, ref):

        logger.info('Ajout d\'un nouvel UCD...')

        args = parser.parse_args()
        v_ucd = args['ucd']

        if len(v_ucd) != 7 and len(v_ucd) != 13:
            return {'message': "L'UCD doit être de longueur 7 ou 13"}, 503

        if not v_ucd.isdigit():
            return {'message': "L'UCD ne doit contenir que des chiffres"}, 503

        try:
            # regarder si on n'a pas déjà une réf associée à cet UCD
            if len(v_ucd) == 7:
                existing_ucd_count = (
                    Ucd.select(Ucd.reference)
                    .where((Ucd.ucd == v_ucd) | (fn.substr(Ucd.ucd, 6, 7) == v_ucd))
                    )

            if len(v_ucd) == 13:
                existing_ucd_count = (
                    Ucd.select(Ucd.reference)
                    .where((Ucd.ucd == v_ucd) | (Ucd.ucd == v_ucd[5:12]))
                    )

            if existing_ucd_count.count() > 0:
                return {'message': 'Cet UCD est associé à la référence %s.' % existing_ucd_count[0].reference}, 503

            Ucd.create(reference=ref, ucd=v_ucd)

            logger.info('Ajout d\'un nouvel UCD... REUSSI')
            return 'Success'

        except Exception as error:
            logger.error(error.args)
            return {'message': error.args}, 503

    def delete(self, ref):
        args = parser.parse_args()
        _ucd = args['ucd']

        logger.info('Suppression d\'un UCD, ref: "%s", UCD: "%s"' % (ref, _ucd))

        try:
            Ucd.delete().where((Ucd.reference == ref) & (Ucd.ucd == _ucd)).execute()
            logger.info('Suppression réussie d\'un UCD, ref: "%s", UCD: "%s"' % (ref, _ucd))
            return 'Success'

        except Exception as error:
            logger.error(error.args)
            return error.args, 503
