from flask_restful import Resource, reqparse
from median.models import Product, Ucd, Cip, Stock, Seuil, Magasin
from peewee import DoesNotExist, fn
import random
import logging

logger = logging.getLogger('median.webserver')

parser = reqparse.RequestParser()
parser.add_argument('ref')
parser.add_argument('designation')
parser.add_argument('ucd')
parser.add_argument('dci')
parser.add_argument('desig_bis')
parser.add_argument('com_med')


class ApiStocks(Resource):

    def post(self, ref=None):

        args = parser.parse_args()
        _ref = args['ref']
        _designation = args['designation']
        _com_med = args['com_med']

        logger.info('Création de produit, ref: "%s"' % (_ref))

        if _ref:
            if not self._isProductReferenceAvailable(_ref):
                logger.warning('La référence existe déjà')
                return {'status': 'Error', 'ref': _ref}
        else:
            logger.info('On crée une nouvelle référence')
            _ref = self._generateProductReference()

        logger.info('On crée le produit en base, ref: "%s", designation: "%s",'
                    ' com_med: "%s"', (_ref, _designation, _com_med))
        Product.create(reference=_ref,
                       designation=_designation,
                       com_med=_com_med)

        return {'status': 'Success', 'ref': _ref}

    def put(self, ref):

        args = parser.parse_args()
        designation = args['designation']
        com_med = args['com_med']

        logger.info('Edition de produit, ref: "%s"' % (ref))

        if not ref:
            logger.warning('Référence manquante!')
            return 'Référence manquante!'
        elif not designation:
            logger.warning('Le champ Désignation ne doit pas être vide!')
            return 'Le champ Désignation ne doit pas être vide!'

        if len(designation) > 100:
            logger.warning('Désignation trop longue!')
            return 'Désignation trop longue!'

        try:
            n = (Product
                 .update({Product.designation: designation,
                          Product.com_med: com_med})
                 .where(Product.reference == ref))
            n.execute()

        except DoesNotExist:
            logger.error('Ce produit n\'existe pas! Réf: "%s"' % (ref))
            return {'message': 'Product does not exist'}, 404
        except Exception as error:
            logger.error(error.args)
            return {'message': error.args}, 503

        logger.info('Le produit a été modifié sans problème. Réf: "%s"' % (ref))
        return 'Success'

    def get(self, ref):

        try:
            p = Product.get(Product.reference == ref)
            stock_info = self._GetProductStockInfo(ref)
            stock_by_magasin = self._getStockByMagasin(ref)
            # ucd_cip_list = self._getProductUCDAndCIPListJSON(ref)

        except DoesNotExist:
            logger.error('Le produit n\'existe pas => Réf: "%s"' % (ref))
            return {'message': 'Product does not exist'}, 404
        except Exception as error:
            logger.error(error.args)
            return {'message': error.args}, 503

        return {
            'id': p.pk,
            'reference': p.reference,
            'designation': p.designation,
            'dci': p.dci,
            'desig_bis': p.desig_bis,
            'com_med': p.com_med,
            'stock': stock_info['stock'],
            'stock_by_mag': stock_by_magasin,
            # 'ucd_cip_list': ucd_cip_list
        }

    def _GetProductStockInfo(self, ref):

        st = (Product
              .select(Product, fn.SUM(Stock.quantite * Stock.fraction / 100).alias('stock'))
              .join(Stock, on=(Product.reference == Stock.reference))
              .where(Product.reference == ref)
              .group_by(Product))

        if not st:
            stock = 0
        else:
            stock = st[0].stock

        s = (Product
             .select(Product,
                     fn.SUM(Seuil.stock_mini).alias('stock_mini'),
                     fn.SUM(Seuil.stock_maxi).alias('stock_maxi'))
             .join(Seuil, on=(Product.reference == Seuil.reference))
             .group_by(Product))

        return {
            'stock': stock,
            'stock_mini': s and s[0].stock_mini or '-',
            'stock_maxi': s and s[0].stock_maxi or '-'
        }

    def _getStockByMagasin(self, ref):

        mags = (Magasin.select(Magasin.mag, Magasin.type_mag)).order_by(Magasin.mag)

        _out = []
        for m in mags:
            _out.append({
                'mag': m.mag,
                'type_mag': m.type_mag,
                'stock': 0,
                'commande': 0,
            })

        return _out

    def _generateProductReference(self):
        while True:
            rand_ref = 'P' + str(round(random.random()*100000000))
            if self._isProductReferenceAvailable(rand_ref):
                break
        return rand_ref

    def _isProductReferenceAvailable(self, ref):
        if (Product.select(Product.reference)
                   .where(Product.reference == ref)
                   .count()) == 0:
            return True
        return False

    def _getProductUCDAndCIPListJSON(self, ref):
        _out = []

        ucd_list = (Ucd.select(Ucd.ucd)
                    .distinct()
                    .where(Ucd.reference == ref)
                    .order_by(Ucd.pk))

        for u in ucd_list:
            ucd_cip_list = Cip.select(Cip.ucd, Cip.cip).distinct().where(Cip.ucd == u.ucd)
            logger.info('Lines : %s.' % len(ucd_cip_list))
            _o = {
                'text': '<b>UCD</b>: ' + u.ucd,
                'children': []
            }
            for c in ucd_cip_list:
                _o['children'].append('<b>CIP</b>: ' + c.cip)
            _out.append(_o)

        return _out
