from flask_restful import reqparse
from flask import session
from common.rest import DataTables
from common.util import logger
from median.models import Stock, Magasin, Adresse, Historique, Gpao, Cip
from peewee import DoesNotExist, fn
import urllib.parse
import datetime
import math

parser = reqparse.RequestParser()
parser.add_argument('product_ref')
parser.add_argument('select')
parser.add_argument('draw', type=int, help='Draw parameter for Datatables')
parser.add_argument('start', type=int, help='Start parameter for Datatables')
parser.add_argument('length', type=int, help='Limit parameter for Datatables')
parser.add_argument('order[0][column]', type=int, help='Ord parameter for DT')
parser.add_argument('order[0][dir]', help='Ord parameter for Datatables')
parser.add_argument("filterByMagasin")
parser.add_argument("form_data")
parser.add_argument("pk")
parser.add_argument("adresse")
parser.add_argument("is_empty_container")
parser.add_argument("service")
parser.add_argument("qte_mvt")
parser.add_argument("lot")
parser.add_argument("date_peremption")
parser.add_argument("contenant")
parser.add_argument("ucd")
parser.add_argument("fraction")

parser.add_argument("quantite_new")
parser.add_argument("quantite_old")
parser.add_argument("lot_new")
parser.add_argument("lot_old")
parser.add_argument("date_peremption_new")
parser.add_argument("date_peremption_old")
parser.add_argument("service")

parser.add_argument("qte_blister_bac")
parser.add_argument("nb_dose_cut")
parser.add_argument("du")
parser.add_argument("qte_prod_blister")


class ApiStock(DataTables):

    def delete(self, ref):
        args = parser.parse_args()
        _pk = args['pk']
        _service = args['service']

        logger.info('Suppression d\'une ligne de stock, pk: "%s"' % (_pk))

        if _service == '-':
            return 'Veuillez indiquer un service', 503

        _padded_adr = self._padAddressField(args['adresse'])

        if not _padded_adr:
            logger.warning('Erreur dans le format adresse')
            return 'Erreur dans le format adresse', 503

        _adrs = Stock.select().where(Stock.adresse == _padded_adr)

        try:
            logger.debug('Suppression de la ligne de stock...')
            Stock.delete().where(Stock.pk == _pk).execute()

            nb_lignes_stocks = (
                Stock.select(Stock.quantite)
                .where(Stock.adresse == _padded_adr)
            )
            if nb_lignes_stocks.count() == 0:
                logger.debug('Mise à jour de l\'adresse, celle-ci devient libre: "%s"' % (_padded_adr))
                Adresse.update({Adresse.etat: 'L'}).where(Adresse.adresse == _padded_adr).execute()

            if _adrs.count() == 0 and args['is_empty_container'] == 'false':
                logger.info('On avance la ou les boîtes qui se trouvaient derrière...')
                self._moveContainerForward(_padded_adr)

            qte_tot = (Stock.select(fn.SUM(Stock.quantite)).where(Stock.reference == ref))

            logger.debug('Ecriture dans la table historique')
            Historique.create(
                chrono=datetime.datetime.now(),
                reference=ref,
                adresse=_padded_adr,
                quantite_mouvement=args['qte_mvt'],
                quantite_totale=qte_tot,
                service=_service,
                type_mouvement='SOR',
                lot=args['lot'],
                pmp=0,
                date_peremption=args['date_peremption'],
                contenant=args['contenant'],
                poste='MEDIAN_CL',
                ucd=args['ucd'].strip(),
                fraction=args['fraction'],
                quantite_picking=qte_tot,
                utilisateur=session['username']
            )

            _calc_id_robot = (Magasin.select(Magasin.id_robot).where(Magasin.mag == _padded_adr[0:3]))
            _id_robot = _calc_id_robot[0].id_robot or 1

            logger.info('GPAO: Ajout d\'un mouvement de type SORTIE.')
            Gpao.create(
                chrono=datetime.datetime.now(),
                etat='A',
                ref=ref,
                qte=args['qte_mvt'],
                lot=args['lot'].strip(),
                type_mvt='S',
                dest=_service,
                tperemp=args['date_peremption'],
                fraction=args['fraction'],
                id_robot=_id_robot
            )

        except Exception as error:
            logger.error(error.args)
            return error.args, 503

        logger.info('Suppression réussie d\'une ligne de stock, pk: "%s"' % (_pk))
        return 'Success'

    def post(self, ref):

        logger.info('Création d\'une ligne de stock...')

        args = parser.parse_args()
        _fd = urllib.parse.parse_qs(args['form_data'])
        _dp = self._convertUserFriendlyDateToDBDateTime(_fd['date_peremption'][0])
        _ucd = _fd['ucd'][0].strip()

        try:
            _calc_fifo = (Stock.select((fn.MAX(Stock.id_fifo)+1).alias('new_fifo')).where(Stock.reference == ref))
            logger.info('Calcul du FIFO : "%s"' % (_calc_fifo))

            _calc_capa = (Cip.select(fn.MIN(Cip.qt_pass).alias('qt_pass')).where(Cip.ucd == '_ucd'))
            _cal = _calc_capa[0].qt_pass or 30
            logger.info('Calcul de la capacité (qt_pass) : "%s"' % (_cal))

            _existing_adr = Stock.select().where((Stock.adresse == _fd['adresse'][0]) & (Stock.reference != ref))
            if len(_existing_adr):
                logger.warning('Cet emplacement est déjà occupé')
                return {'message': 'Cet emplacement est déjà occupé'}, 503

            logger.debug('Ajout ligne de stock, dans la table f_stock')
            Stock.create(adresse=_fd['adresse'][0],
                         reference=ref,
                         quantite=_fd['quantite'][0],
                         id_fifo=_calc_fifo[0].new_fifo,
                         lot=_fd['lot'][0],
                         date_peremption=_dp,
                         date_entree=str(datetime.datetime.now()).split('.')[0],
                         contenant=_fd['contenant'][0],
                         magasin=_fd['magasin'][0],
                         ucd=_ucd,
                         fraction=_fd['fraction'][0],
                         capa=_cal,
                         )

            logger.debug('Mise à jour adresse "%s", état O & code contenant: "%s"' % (
                _fd['adresse'][0], _fd['contenant'][0]))
            (Adresse.update({Adresse.etat: 'O', Adresse.contenant: _fd['contenant'][0]})
                .where(Adresse.adresse == _fd['adresse'][0]).execute())

            qte_tot = (Stock.select(fn.SUM(Stock.quantite)).where(Stock.reference == ref))

            logger.debug('Ecriture dans la table historique')
            Historique.create(
                chrono=datetime.datetime.now(),
                reference=ref,
                adresse=_fd['adresse'][0],
                quantite_mouvement=_fd['quantite'][0],
                quantite_totale=qte_tot,
                quantite_picking=qte_tot,
                service=_fd['service'][0],
                type_mouvement='ENT',
                lot=_fd['lot'][0],
                pmp=0,
                date_peremption=_dp,
                contenant=_fd['contenant'][0],
                poste='MEDIAN_CL',
                ucd=_ucd,
                fraction=_fd['fraction'][0],
                utilisateur=session['username']
            )

            _calc_id_robot = (Magasin.select(Magasin.id_robot).where(Magasin.mag == _fd['magasin'][0]))
            _id_robot = _calc_id_robot[0].id_robot or 1

            logger.info('GPAO : Ajout d\'un mouvement de type ENTREE')
            Gpao.create(
                chrono=datetime.datetime.now(),
                etat='A',
                ref=ref,
                qte=_fd['quantite'][0],
                lot=_fd['lot'][0],
                type_mvt='E',
                dest=_fd['service'][0],
                tperemp=_dp,
                fraction=_fd['fraction'][0],
                id_robot=_id_robot
            )

        except Exception as error:
            logger.error(error.args)
            return {'message': error.args}, 503

        logger.info('Création d\'une ligne de stock... REUSSI')
        return 'Success'

    def _put_new_quantite(self, pk, adresse, ref, quantite_new, quantite_old, lot, peremp,
                          contenant, ucd, fraction, service, _du, _qte_prod_blister):

        logger.info('Edition de la quantité d\'une ligne de stock...')

        _diff = quantite_new - quantite_old
        if (quantite_new == 0):
            logger.debug('Nouvelle quantité = 0, donc suppression dans la table stock et '
                         'mise à jour de l\'adresse "%s"' % (adresse))
            Stock.delete().where(Stock.pk == pk).execute()
            (Adresse.update({Adresse.etat: 'L'}).where(Adresse.adresse == adresse).execute())
        else:
            logger.debug('Edition de la quantité associéer à une ligne de stock,'
                         ' pk: "%s", qté: "%s"' % (pk, quantite_new))

            qte_blister_bac = 0
            nb_dose_cut = 0

            if (_du == '1'):
                qte_blister_bac = math.ceil(int(quantite_new) / int(_qte_prod_blister))
                nb_dose_cut = int(quantite_new) % int(_qte_prod_blister)
                if (nb_dose_cut == 0):
                    nb_dose_cut = _qte_prod_blister

            Stock.update({
                Stock.quantite: quantite_new, Stock.qte_blister_bac: qte_blister_bac,
                Stock.nb_dose_cut: nb_dose_cut}).where((Stock.pk == pk)).execute()

        qte_tot = (Stock.select(fn.SUM(Stock.quantite)).where(Stock.reference == ref))

        logger.debug('Ecriture dans la table historique')
        Historique.create(
            chrono=datetime.datetime.now(),
            reference=ref,
            adresse=adresse,
            quantite_mouvement=_diff,
            quantite_totale=qte_tot,
            service=service,
            type_mouvement='INV',
            lot=lot,
            pmp=0,
            date_peremption=peremp,
            contenant=contenant,
            poste='MEDIANWEB',
            ucd=ucd,
            fraction=fraction,
            utilisateur=session['username']
        )

        _calc_id_robot = (Magasin.select(Magasin.id_robot).where(Magasin.mag == adresse[0:3]))
        _id_robot = _calc_id_robot[0].id_robot or 1

        # logger.info('GPAO: Ajout d\'un mouvement de type inventaire (on supprime l\'ancienne quantité)')
        # Gpao.create(
        #     chrono=datetime.datetime.now(),
        #     etat='A',
        #     ref=ref,
        #     qte=-quantite_old,
        #     lot=lot,
        #     type_mvt='I',
        #     dest=service,
        #     tperemp=peremp,
        #     fraction=fraction,
        #     id_robot=_id_robot
        # )

        logger.info('GPAO: Ajout d\'un mouvement de type inventaire (on ajoute la nouvelle quantité)')
        Gpao.create(
            chrono=datetime.datetime.now(),
            etat='A',
            ref=ref,
            qte=quantite_new-quantite_old,
            lot=lot,
            type_mvt='I',
            dest=service,
            tperemp=peremp,
            fraction=fraction,
            id_robot=_id_robot
        )

        logger.info('Edition de la quantité d\'une ligne de stock... REUSSI')

    def put(self, ref):

        logger.info('Edition d\'une ligne de stock...')

        args = parser.parse_args()
        _pk = args['pk']
        _qte_n = args['quantite_new']
        _qte_o = args['quantite_old']
        # _lot_n = args['lot_new']
        _lot_o = args['lot_old']
        # _dp_n = self._convertUserFriendlyDateToDBDateTime(args['date_peremption_new']).split(' ')[0]
        _dp_o = args['date_peremption_old']
        _serv = args['service']
        _adr = self._padAddressField(args['adresse'])
        _contenant = args['contenant']
        _ucd = args['ucd']
        _fraction = args['fraction']
        _du = args['du']
        _qte_prod_blister = args['qte_prod_blister']

        try:
            if _qte_n != _qte_o:
                self._put_new_quantite(_pk, _adr, ref, int(_qte_n), int(_qte_o), _lot_o,
                                       _dp_o, _contenant, _ucd, _fraction, _serv, _du, _qte_prod_blister)
            # if _lot_n != _lot_o:
            #     self._put_new_lot(_pk, _adr, ref, int(_qte_n), int(_qte_o), _lot_n, _lot_o, _dp_n,
            #                       _dp_o, _contenant, _ucd, _fraction, _serv)
            # if _dp_n != _dp_o:
            #     self._put_new_date_peremption(_pk, _adr, ref, int(_qte_n), int(_qte_o), _lot_n,
            #                                   _dp_n, _dp_o, _contenant, _ucd, _fraction, _serv)
        except Exception as error:
            logger.error(error.args)
            return {'message': error.args}, 503

        logger.info('Edition d\'une ligne de stock... REUSSI')
        return 'Success'

    def get(self, ref):
        args = parser.parse_args()
        v_draw = args['draw']
        v_start = args['start']
        v_length = args['length']
        v_orderby = args['order[0][column]']
        v_dir = args['order[0][dir]']
        v_filter_by_magasin = args['filterByMagasin']

        try:
            if not v_filter_by_magasin:
                total_stocks_count = (
                    Stock.select(
                        Stock.pk,
                        Stock.bloque,
                        Stock.quantite)
                    .where(Stock.reference == ref)
                    .count())
                filtered_stocks_query = (
                    Stock.select().where(Stock.reference == ref))
            else:
                ms = v_filter_by_magasin.split(",")
                total_stocks_count = (
                    Stock.select(Stock.pk, Stock.bloque, Stock.quantite)
                    .where((Stock.reference == ref) & (Stock.magasin << ms))
                    .count())
                filtered_stocks_query = (
                    Stock.select().where((Stock.reference == ref) & (Stock.magasin << ms)))

            if (v_orderby == 1):
                v_orderby_column = Stock.bloque
            elif (v_orderby == 2):
                v_orderby_column = Stock.adresse
            elif (v_orderby == 3):
                v_orderby_column = Stock.quantite
            elif (v_orderby == 4):
                v_orderby_column = Stock.fraction
            elif (v_orderby == 5):
                v_orderby_column = Stock.ucd
            elif (v_orderby == 6):
                v_orderby_column = Stock.date_sortie
            elif (v_orderby == 7):
                v_orderby_column = Stock.date_entree
            elif (v_orderby == 8):
                v_orderby_column = Stock.lot
            elif (v_orderby == 9):
                v_orderby_column = Stock.date_peremption
            elif (v_orderby == 10):
                v_orderby_column = Stock.contenant
            else:
                v_orderby_column = Stock.adresse

            if (v_dir == 'desc'):
                v_orderby_column_dir = v_orderby_column.desc()
            else:
                v_orderby_column_dir = v_orderby_column

            filtered_stocks = (filtered_stocks_query
                               .order_by(v_orderby_column_dir))

            paged_stocks = filtered_stocks.limit(v_length).offset(v_start)

            logger.debug('Lines : %s.' % len(paged_stocks))

            return self.render([{
                'pk': s.pk,
                'bloque': ('OUI' if s.bloque else 'NON'),
                'emplacement': s.adresse,
                'quantite': s.quantite,
                'ucd': s.ucd,
                'date_sortie': str(s.date_sortie or '-'),
                'date_entree': str(s.date_entree or '-'),
                'lot': s.lot,
                'date_peremption': str(s.date_peremption).split(' ')[0],
                'contenant': s.contenant,
                'fraction': s.fraction
            } for s in paged_stocks],
                v_draw,
                total_stocks_count,
                filtered_stocks.count())

        except DoesNotExist:
            logger.error('Get stock Datatables raised a DoesNotExist exception')
            return self.render([], v_draw, 0, 0)
        except Exception as error:
            logger.error('Get stock Datatables raised an exception: ', error.args)
            return self.render([], v_draw, 0, 0, error.args)

    def ecoBasicDecodeURIComponent(self, str):
        return str.replace('__', ' ').replace('--', '/')

    def _convertUserFriendlyDateToDBDateTime(self, ufDate):
        _dl = ufDate.split('-')
        if len(_dl[0]) == 4:
            return ufDate + ' 00:00:00'
        return _dl[2] + '-' + _dl[1] + '-' + _dl[0] + ' 00:00:00'

    def _padAddressField(self, adr):
        _adr_items = adr.split('.')
        # if len(_adr_items) != 5:
        #     return False
        _o = []
        for a in _adr_items:
            _o.append(a.rjust(3))

        return '.'.join(_o)

    def _moveContainerForward(self, adr):
        _els = adr.split('.')
        _pos = int(_els[-1].lstrip())
        if _pos > 1:
            return False

        _back_adrs = []
        _els[-1] = '  2'
        _back_adrs.append('.'.join(_els))
        _els[-1] = '  3'
        _back_adrs.append('.'.join(_els))

        _a = (Adresse.select(Adresse.contenant)
                     .where((Adresse.adresse == _back_adrs[0]) | (Adresse.adresse == _back_adrs[1])))

        _cont = '' if _a.count() == 0 else _a[0].contenant

        (Adresse.update({Adresse.contenant: _cont, Adresse.etat: 'O'})
            .where(Adresse.adresse == adr).execute())

        (Adresse.update({Adresse.contenant: '', Adresse.etat: 'L'})
            .where((Adresse.adresse == _back_adrs[0]) | (Adresse.adresse == _back_adrs[1]))
            .execute())

        (Stock.update({Stock.adresse: adr})
            .where((Stock.adresse == _back_adrs[0]) | (Stock.adresse == _back_adrs[1]))
            .execute())
