from flask_restful import Resource, reqparse
# from flask import session
from median.models import Seuil, Stock, Magasin, Dispensation, DispensationItem, Zone, ZoneMag
from peewee import DoesNotExist, fn
import logging

logger = logging.getLogger('median.webserver')

parser = reqparse.RequestParser()
parser.add_argument('product_ref')
parser.add_argument('select')
parser.add_argument('draw', type=int, help='Draw parameter for Datatables')
parser.add_argument('start', type=int, help='Start parameter for Datatables')
parser.add_argument('length', type=int, help='Limit parameter for Datatables')
parser.add_argument('order[0][column]', type=int, help='Ord parameter for DT')
parser.add_argument('order[0][dir]', help='Ord parameter for Datatables')
parser.add_argument('mag')
parser.add_argument('fraction')
parser.add_argument('seuil_min')
parser.add_argument('seuil_max')
parser.add_argument('quantite')
parser.add_argument('en_commande')
parser.add_argument('pk')


class DataTables(Resource):
    def render(self, datas, draw, recordsTotal, recordsFiltered, error=None):
        o = {
            'draw': draw,
            'recordsTotal': recordsTotal,
            'recordsFiltered': recordsFiltered,
            'data': datas
        }
        if error is not None:
            o['error'] = error
        return o


class ApiSeuils(DataTables):

    def put(self, ref):
        args = parser.parse_args()
        _pk = args['pk']
        _seuil_min = args['seuil_min']
        _seuil_max = args['seuil_max']

        logger.info('Edition d\'un seuil, pk: "%s"' % (_pk))

        try:
            Seuil.update({Seuil.stock_mini: _seuil_min, Seuil.stock_maxi: _seuil_max}).where(Seuil.pk == _pk).execute()

        except Exception as error:
            logger.error(error.args)
            return error.args, 503

        logger.info('Edition réussie d\'un seuil, pk: "%s"' % (_pk))
        return 'Success'

    def delete(self, ref):
        args = parser.parse_args()
        _pk = args['pk']

        logger.info('Suppression d\'un seuil, pk: "%s"' % (_pk))

        try:
            Seuil.delete().where(Seuil.pk == _pk).execute()

        except Exception as error:
            logger.error(error.args)
            return error.args, 503

        logger.info("""Suppression réussie d'un seuil, pk: "%s" """ % (_pk))
        return 'Success'

    def post(self, ref):

        logger.info("Ajout d'un seuil stock produit...")

        args = parser.parse_args()
        _mag = args['mag']
        _fraction = args['fraction']
        _seuil_min = args['seuil_min']
        _seuil_max = args['seuil_max']

        try:
            # regarder si on n'a pas déjà une ligne de seuils pour ce couple ref/fraction
            meme_seuils_count = (
                Seuil.select()
                .where((Seuil.reference == ref) & (Seuil.fraction == _fraction) & (Seuil.zone == _mag))
                .count())
            if meme_seuils_count > 0:
                return {'message': 'Il y a déjà un seuil pour ce magasin et pour cette fraction.'}, 503

            (Seuil.create(
                reference=ref,
                zone=_mag,
                stock_mini=int(_seuil_min),
                stock_maxi=int(_seuil_max),
                fraction=int(_fraction)))

        except Exception as error:
            logger.error(error.args)
            return {'message': error.args}, 503

        logger.info('Ajout d\'un seuil stock produit... REUSSI')
        return 'Success'

    def get(self, ref):
        args = parser.parse_args()
        v_draw = args['draw']
        v_orderby = args['order[0][column]']
        v_dir = args['order[0][dir]']

        try:
            total_seuils_count = (
                Seuil.select()
                .where(Seuil.reference == ref)
                .count())
            filtered_seuils_query = (
                Seuil.select().where(Seuil.reference == ref))

            if (v_orderby == 1):
                v_orderby_column = Seuil.zone
            elif (v_orderby == 2):
                v_orderby_column = Seuil.fraction
            elif (v_orderby == 3):
                v_orderby_column = Seuil.stock_mini
            elif (v_orderby == 4):
                v_orderby_column = Seuil.stock_maxi
            else:
                v_orderby_column = Seuil.zone

            if (v_dir == 'desc'):
                v_orderby_column_dir = v_orderby_column.desc()
            else:
                v_orderby_column_dir = v_orderby_column

            filtered_seuils = (filtered_seuils_query
                               .order_by(v_orderby_column_dir))

            logger.debug('Lines : %s.' % len(filtered_seuils))

            return self.render([{
                'pk': s.pk,
                'mag': s.zone,
                'fraction': s.fraction,
                'seuil_min': s.stock_mini,
                'seuil_max': s.stock_maxi,
                'quantite': self._get_seuil_quantite(ref, s.fraction, s.zone),
                'commande': self._get_seuil_commande(ref, s.fraction, s.stock_maxi, s.zone),
                'en_commande': (0 if self._get_seuil_quantite(ref, s.fraction, s.zone) +
                                self._get_seuil_commande(ref, s.fraction, s.stock_maxi, s.zone) >= s.stock_mini
                                else s.stock_maxi - self._get_seuil_quantite(ref, s.fraction, s.zone) +
                                self._get_seuil_commande(ref, s.fraction, s.stock_maxi, s.zone))

            } for s in filtered_seuils],
                v_draw,
                total_seuils_count,
                filtered_seuils.count())

        except DoesNotExist:
            logger.error('Get Seuils raised a DoesNotExist exception')
            return self.render([], v_draw, 0, 0)
        except Exception as error:
            logger.error('Get Seuils raised an exception: ', error.args)
            return self.render([], v_draw, 0, 0, error.args)

    def _get_seuil_quantite(self, ref, fraction, poste):
        _mag = self._get_mag(poste)
        if _mag == '-':
            return '-'
        _q = (
                Stock.select(fn.SUM(Stock.quantite).alias('qte'))
                .where((Stock.reference == ref) & (Stock.fraction == fraction) & (Stock.magasin == _mag)))

        stk = Stock().select(fn.SUM(
            Stock.quantite).alias("total")).join(ZoneMag, on=(Stock.magasin == ZoneMag.mag)).join(
            Zone, on=((Zone.zone == ZoneMag.zone) & (Zone.appro == 1))).where(
            Stock.zone_admin == poste, Stock.reference == ref, Stock.fraction == fraction)

        return ((_q[0].qte or 0) + (stk[0].total or 0))

    def _get_seuil_commande(self, ref, fraction, stock_maxi, poste):
        _mag = self._get_mag(poste)
        if _mag == '-':
            return '-'

        # calcul des quantités déjà en commande
        q = (Dispensation.select(
            (fn.SUM(DispensationItem.qte_dem - DispensationItem.qte_serv)).alias('cmd')).join(
                    DispensationItem, on=(
                        (Dispensation.liste == DispensationItem.liste)
                        & (Dispensation.mode == DispensationItem.mode)
                        & (fraction == DispensationItem.fraction)
                        & (DispensationItem.reference == ref)))
                .where(  # noqa
                        (Dispensation.mode == 'E')
                        & (Dispensation.zone_fin == poste))
                .group_by(Dispensation.zone_fin))

        qty_in_list = 0 if not q else q[0].cmd
        if qty_in_list < 0:
            qty_in_list = 0
        return qty_in_list

    def _get_mag(self, type_mag):
        _m = Magasin.select(Magasin.mag).where(Magasin.type_mag == type_mag)
        if len(_m) == 0:
            return '-'
        return _m[0].mag
