from flask_restful import Resource, reqparse
from flask import jsonify
from median.models import FListe, Magasin, Config, Endpoint, FItem, Stock
from median.constant import TypeListe, TypeServiListe, EtatListe, RIEDL_PRIORITY
# from median.views import RawConfig
from common.util import logger
from peewee import DoesNotExist
from playhouse.shortcuts import model_to_dict
from common.status import HTTP_400_BAD_REQUEST, HTTP_404_NOT_FOUND
import requests
import datetime


class ApiRiedlListesSorties(Resource):

    def get(self, liste=None):
        logger.info("Récupérer les listes de sorties")
        if liste is None:
            return {
                'listes': self._getRiedlEntryListJSON(),
            }
        else:
            return self._get_liste(liste)

    def delete(self):
        """Delete list by name"""
        parser = reqparse.RequestParser()
        parser.add_argument('liste', type=str, help='List name')
        args = parser.parse_args()
        list_name = args['liste']
        # Check if list is relaed to a Riedl
        try:
            lst = FListe.get(FListe.mode == TypeListe.Output.value, FListe.liste == list_name)
            # Delete a liste, delete the items
            lst.delete_instance()

        except DoesNotExist:
            logger.error('Liste %s does not exists' % (list_name,))
            return {'message': 'Liste %s does not exists' % (list_name,)}, HTTP_404_NOT_FOUND

        return {'liste': args['liste']}

    def patch(self, liste=None):
        """Update value on f_liste"""

        parser = reqparse.RequestParser()
        parser.add_argument('liste', required=True, type=str, help='List name')
        parser.add_argument('field', required=True, type=str, help='Name of the field')
        parser.add_argument('type', type=str, help='Type of the field')
        parser.add_argument('value', required=True, type=str, help='Value of the field')
        args = parser.parse_args()

        tmp_type = args.get('type', 'str')
        tmp_val = args['value']
        if tmp_type == 'bool':
            if tmp_val in ('0', 'false'):
                tmp_val = False
            else:
                tmp_val = True

        list_name = args['liste']
        try:
            lst = FListe.get(FListe.mode == TypeListe.Output.value, FListe.liste == list_name)
            getattr(lst, args['field'])
            setattr(lst, args['field'], tmp_val)
            lst.save()

        except DoesNotExist:
            logger.error('Liste %s does not exists' % (list_name,))
            return {'message': 'Liste %s does not exists' % (list_name,)}, HTTP_404_NOT_FOUND
        except AttributeError:
            logger.error('Field  %s not exists on f_liste' % (args['field'],))
            return {'message': 'Field  %s not exists on f_liste' % (args['field'],)}, HTTP_400_BAD_REQUEST

        return {'result': 'OK', 'type': tmp_type, 'val': tmp_val}

    def _get_liste(self, list_name):
        """Retrieve the list information"""
        res = {}
        try:
            lst = FListe.get(FListe.mode == TypeListe.Output.value, FListe.liste == list_name)
            endpoints = Endpoint.select(Endpoint).where(Endpoint.type_dest == 'RIEDL', Endpoint.secteur == lst.zone_fin)
            riedls = Magasin.select(Magasin).where(Magasin.eco_type == 'L')

            res['data'] = model_to_dict(lst)
            res['selection'] = {}
            res['selection']['endpoint'] = [model_to_dict(enp) for enp in endpoints]
            res['selection']['riedl'] = [model_to_dict(rdl) for rdl in riedls]
            res['selection']['priority'] = [{'value': i[0], 'name': i[1]} for i in RIEDL_PRIORITY]
            return jsonify(res)
        except DoesNotExist:
            res['error'] = {'message': 'List %s does not exists' % list_name}
            return res, HTTP_404_NOT_FOUND
        return {}, HTTP_400_BAD_REQUEST

    def _getRiedlEntryListJSON(self):
        res = []
        # First we search all entry list not affected to a riedl
        res.append({
            'text': '<b>Globale</b>',
            'id': 'outglobal',
            'children': self._getRiedlGlobaleList(),
        })
        res.append({
            'text': '<b>Nominative</b>',
            'id': 'outnominative',
            'children': self._getRiedlNominativeList(),
        })
        res.append({
            'text': '<b>Retrait de Lot</b>',
            'id': 'outlot',
            'children': self._getRiedlRetraitLotList(),
        })
        res.append({
            'text': '<b>Périmé</b>',
            'id': 'outperime',
            'children': self._getRiedlPerimeList(),
        })
        return res

    def _getRiedlGlobaleList(self):
        res = []
        mags = Magasin.select(Magasin).where(Magasin.eco_type == 'L')
        for m in mags:
            ul = {
                'text': '<b>%s</b>' % m.libelle,
                'id': 'global-%i' % (m.pk,),
                'children': []
            }
            # Search all lists affect to this riedl
            rdl_list = FListe.select(FListe).where(
                FListe.mode == TypeListe.Output.value,
                FListe.type_servi << [TypeServiListe.GlobaleBoite.value, TypeServiListe.RiedlBoite.value],
                FListe.zone_deb == 'RIEDL', FListe.zone_fin == m.type_mag).order_by(FListe.service)
            # print(rdl_list)
            for lst in rdl_list:
                li = {
                    'text': '%s-(%s)' % (lst.liste, lst.etat),
                    'id': 'model-%s' % (lst.liste,),
                    'icon': 'jstree-file',
                    'li_attr': {'class': 'color:black, background-color: blue'}
                }
                if lst.etat == 'S':
                    li['text'] = '<del>%s-(%s)</del>' % (lst.liste, lst.etat)
                elif lst.etat == 'E':
                    li['text'] = '<mark>%s-(%s)</mark>' % (lst.liste, lst.etat)
                ul['children'].append(li)
            res.append(ul)
        return res

    def _getRiedlNominativeList(self):
        res = []
        mags = Magasin.select(Magasin).where(Magasin.eco_type == 'L')
        for m in mags:
            ul = {
                'text': '<b>%s</b>' % m.libelle,
                'id': 'nominat-%i' % (m.pk,),
                'children': []
            }
            # Search all lists affect to this riedl
            rdl_list = FListe.select(FListe).where(
                FListe.mode == TypeListe.Output.value,
                FListe.type_servi == TypeServiListe.Nominatif.value,
                FListe.zone_deb == 'RIEDL', FListe.zone_fin == m.type_mag).order_by(FListe.service)
            # print(rdl_list)
            for lst in rdl_list:
                li = {
                    'text': '%s' % (lst.liste,),
                    'id': 'model-%s' % (lst.liste,),
                    'icon': 'jstree-file'
                }
                if lst.etat == 'S':
                    li['text'] = '<del>%s-(%s)</del>' % (lst.liste, lst.etat)
                elif lst.etat == 'E':
                    li['text'] = '<mark>%s-(%s)</mark>' % (lst.liste, lst.etat)
                ul['children'].append(li)
            res.append(ul)
        return res

    def _getRiedlRetraitLotList(self):
        res = []
        mags = Magasin.select(Magasin).where(Magasin.eco_type == 'L')
        for m in mags:
            ul = {
                'text': '<b>%s</b>' % m.libelle,
                'id': 'retlot-%i' % (m.pk,),
                'children': []
            }
            # Search all lists affect to this riedl
            rdl_list = FListe.select(FListe).where(
                FListe.mode == TypeListe.Output.value,
                FListe.type_servi == TypeServiListe.RetraitLot.value,
                FListe.zone_deb == 'RIEDL', FListe.zone_fin == m.type_mag).order_by(FListe.service)
            # print(rdl_list)
            for lst in rdl_list:
                li = {
                    'text': '%s' % (lst.liste,),
                    'id': 'model-%s' % (lst.liste,),
                    'icon': 'jstree-file'
                }
                if lst.etat == 'S':
                    li['text'] = '<del>%s-(%s)</del>' % (lst.liste, lst.etat)
                elif lst.etat == 'E':
                    li['text'] = '<mark>%s-(%s)</mark>' % (lst.liste, lst.etat)
                ul['children'].append(li)
            res.append(ul)
        return res

    def _getRiedlPerimeList(self):
        res = []
        mags = Magasin.select(Magasin).where(Magasin.eco_type == 'L')
        for m in mags:
            ul = {
                'text': '<b>%s</b>' % m.libelle,
                'id': 'perime-%i' % (m.pk,),
                'children': []
            }
            # Search all lists affect to this riedl
            rdl_list = FListe.select(FListe).where(
                FListe.mode == TypeListe.Output.value,
                FListe.type_servi == TypeServiListe.Perimee.value,
                FListe.zone_deb == 'RIEDL', FListe.zone_fin == m.type_mag).order_by(FListe.service)
            # print(rdl_list)
            for lst in rdl_list:
                li = {
                    'text': '%s' % (lst.liste,),
                    'id': 'model-%s' % (lst.liste,),
                    'icon': 'jstree-file'
                }
                if lst.etat == 'S':
                    li['text'] = '<del>%s-(%s)</del>' % (lst.liste, lst.etat)
                elif lst.etat == 'E':
                    li['text'] = '<mark>%s-(%s)</mark>' % (lst.liste, lst.etat)
                ul['children'].append(li)
            res.append(ul)
        return res


class ApiRiedlException(Exception):
    pass


class ApiRiedlCommandSorties(Resource):

    def get(self):
        return "1"

    def post(self):
        # REcuperation de l'i de la commande
        parser = reqparse.RequestParser()
        parser.add_argument('liste', type=str, help='List name')
        args = parser.parse_args()
        liste = args['liste']
        res = {
            'liste': liste
        }

        # Récupération de la liste pour savoir sur quel RIEDL la lancer
        try:
            lst = FListe.get(
                FListe.mode == TypeListe.Output.value,
                FListe.liste == liste,
                FListe.zone_deb == 'RIEDL')
            res['riedl'] = lst.zone_fin
            res['exit'] = lst.no_pilulier
            res['priority'] = 1
        except DoesNotExist:
            logger.error('list %s for rield does not exists' % (liste,))
            return {'message': 'list %s for rield does not exists' % (liste,)}, HTTP_404_NOT_FOUND

        # Check if wee need to split the list
        try:
            if lst.valide_sel and lst.selectionne == 0:
                lst_prelev = self._split_liste(lst)
                if lst_prelev is not None:
                    # we launch the unload of this list
                    self._launch_liste(lst_prelev.zone_fin, lst_prelev)

            # We launch teh standard list
            self._launch_liste(lst.zone_fin, lst)
        except ApiRiedlException:
            logger.error('dispatcher URL for rield %s does not exists' % (lst.zone_fin,))
            return {'message': 'dispatcher URL for rield %s does not exists' % (lst.zone_fin,)}, HTTP_404_NOT_FOUND

        return res

    def _launch_liste(self, r_poste, list_obj):
        # Récupération de l'URL du dispatcher
        try:
            cfg = Config.get(poste=r_poste, cle='cfg', propriete='k_rdl_dispatcher_url')
        except DoesNotExist:
            logger.error('dispatcher URL for rield %s does not exists' % (r_poste,))
            raise ApiRiedlException('dispatcher URL for rield %s does not exists' % (r_poste,))

        # Envoi de la requete avec requests
        try:
            url = cfg.value
            if url.endswith('/'):
                url = url[:-1]
            requests.post('%s/RiedlListUpdate' % (url,), json={'id': list_obj.pk}, timeout=10)
        except requests.exceptions.RequestException as e:
            logger.error('Call dispatcher  %s failed' % (cfg.value,))
            return {'message': 'Call dispatcher  %s failed' % (cfg.value,), 'trace': str(e)}, HTTP_404_NOT_FOUND

    def _split_liste(sel, list_obj):
        """SPlit the list in 2"""
        logger.info("Creation liste de prelevement")
        item_list = []

        # Retrieve the taking mode endpoint for this endpoint
        try:
            endp = Endpoint.get(type_dest='RIEDL', secteur=list_obj.zone_fin, type_peigne=list_obj.no_pilulier)
            logger.info("Bouche de sortie standard: %s" % endp.code)
            if endp.adr1:
                # If endpoint taking mode not defined, we cannot create the list
                # We can creat an f_liste_error, to said not taking mode endpoint defined !
                logger.error("Pas de bouche de sortie de prélèvement, pour %s" % endp.code)
                return None
            endp_prelev = Endpoint.get(type_dest='RIEDL', code=endp.adr1)
            logger.info("Bouche de sortie pour le prélèvement: %s" % endp_prelev.code)
        except DoesNotExist:
            # TODO: if endpoint not exists, we quit or raise and error
            return None

        lst2 = FListe()
        lst2.mode = list_obj.mode
        lst2.liste = list_obj.liste + "-P"
        lst2.liste_origine = list_obj.liste
        lst2.valide_sel = False
        lst2.etat = EtatListe.Vierge.value
        lst2.type_servi = TypeServiListe.GlobaleBoite.value
        lst2.fusion = "PRELEVEMENT"
        lst2.service = list_obj.service
        lst2.num_sej = list_obj.num_sej
        lst2.ipp = list_obj.ipp
        lst2.num_ipp = list_obj.num_ipp
        lst2.zone_deb = list_obj.zone_deb
        lst2.zone_fin = list_obj.zone_fin
        lst2.pos_pilulier = 1
        lst2.no_pilulier = endp_prelev.type_peigne
        lst2.nb_item = 1
        lst2.selectionne = 1

        # Retrieve the warehouse for this riedl
        try:
            mag = Magasin.get(Magasin.type_mag == list_obj.zone_fin, Magasin.eco_type == 'L')
        except DoesNotExist:
            # TODO: THis case is not possible
            pass

        # For each line, compute line with stock and check if necessary to split the list
        itms = FItem.select(FItem).where(FItem.liste == list_obj.liste, FItem.mode == list_obj.etat)
        line = 1
        for i in itms:
            cpt_qte = 0
            # old_qte = 0
            # check quantity to unload (only unblock stock line)
            stk = Stock.select(Stock).where(Stock.magasin == mag, Stock.reference == i.reference,
                                            Stock.bloque == 0).order_by(Stock.date_peremption)
            for s in stk:
                cpt_qte += s.quantite

                if i.qte_dem == cpt_qte:
                    # If quantity is the same, we move to the next item
                    break
                elif i.qte_dem > cpt_qte:
                    # Not enougth quantity we contenue with another stock.
                    continue
                else:
                    new_item = FItem()
                    new_item.liste = lst2.liste
                    new_item.mode = lst2.mode
                    new_item.etat = EtatListe.Vierge.value
                    new_item.dest = lst2.service
                    new_item.fraction = 100
                    new_item.type_servi = lst2.type_servi
                    new_item.reference = i.reference
                    new_item.item = str(line).zfill(6)
                    new_item.qte_dem = cpt_qte - i.qte_dem      # Quantité a retournée.
                    new_item.qte_serv = 0
                    new_item.type_servi = lst2.type_servi
                    new_item.dtprise = datetime.datetime.now()
                    new_item.num_ipp = lst2.num_ipp
                    new_item.num_sej = lst2.num_sej

                    item_list.append(new_item)

                    line += 1

                    # TODO: after we remove the quantity on the original list
                    i.qte_serv = cpt_qte - i.qte_dem        # on indique la quantité retournéé comme deja prise
                    i.save()
        # If items on the list, wee
        if item_list:
            lst2.nb_item = line
            lst2.save()
            for itm in item_list:
                itm.save()

            return lst2

        return None
