from flask_restful import Resource, reqparse
from median.models import Product, Ucd, Cip, Stock, Seuil, FListe, FItem, Magasin
from peewee import DoesNotExist, fn, JOIN
from common.status import HTTP_400_BAD_REQUEST, HTTP_404_NOT_FOUND, HTTP_503_SERVICE_UNAVAILABLE
import random
import logging

logger = logging.getLogger('median.webserver')

parser = reqparse.RequestParser()
parser.add_argument('ref')
parser.add_argument('designation')
parser.add_argument('ucd')
parser.add_argument('dci')
parser.add_argument('desig_bis')
parser.add_argument('com_med')
parser.add_argument('risk')
parser.add_argument('narcotic')


class ApiProduct(Resource):

    def post(self, ref=None):

        args = parser.parse_args()
        _ref = args['ref']
        _designation = args['designation']
        _com_med = args['com_med']

        logger.info('Création de produit, ref: "%s"' % (_ref))

        if _ref:
            if not self._isProductReferenceAvailable(_ref):
                logger.warning('La référence existe déjà')
                return {'status': 'Error', 'ref': _ref}
        else:
            logger.info('On crée une nouvelle référence')
            _ref = self._generateProductReference()

        logger.info('On crée le produit en base, ref: "%s", designation: "%s",'
                    ' com_med: "%s"', (_ref, _designation, _com_med))
        Product.create(reference=_ref,
                       designation=_designation,
                       com_med=_com_med)

        return {'status': 'Success', 'ref': _ref}

    def put(self, ref):

        args = parser.parse_args()
        designation = args.get('designation', '')
        com_med = args.get('com_med', '')
        risk = args.get('risk', False)
        narco = args.get('narcotic', False)

        logger.info('Edition de produit, ref: "%s"' % (ref))

        if not ref:
            logger.warning('Référence manquante!')
            return {'message': 'Référence manquante!'}, HTTP_400_BAD_REQUEST
        elif not designation:
            logger.warning('Le champ Désignation ne doit pas être vide!')
            return {'message': 'Le champ Désignation ne doit pas être vide!'}, HTTP_400_BAD_REQUEST

        if len(designation) > 100:
            logger.warning('Désignation trop longue!')
            return {'message': 'Désignation trop longue!'}, HTTP_400_BAD_REQUEST

        try:
            Product.get(Product.reference == ref)
            n = (Product.update({
                    Product.designation: designation,
                    Product.com_med: com_med,
                    Product.risque: risk,
                    Product.stup: narco
                }).where(Product.reference == ref))
            n.execute()

        except DoesNotExist:
            logger.error('Ce produit n\'existe pas! Réf: "%s"' % (ref))
            return {'message': 'Product does not exist'}, HTTP_404_NOT_FOUND
        except Exception as error:
            logger.error(error.args)
            return {'message': error.args}, HTTP_503_SERVICE_UNAVAILABLE

        logger.info('Le produit a été modifié sans problème. Réf: "%s"' % (ref))
        return 'Success'

    def get(self, ref):

        try:
            p = Product.get(Product.reference == ref)
            stock_info = self._GetProductStockInfo(ref)
            stock_by_magasin = self._getStockByMagasin(ref)
            ucd_cip_list = self._getProductUCDAndCIPListJSON(ref)

        except DoesNotExist:
            logger.error("""Le produit n'existe pas => Réf: "%s" """ % (ref))
            return {'message': 'Product does not exist'}, HTTP_404_NOT_FOUND
        except Exception as error:
            logger.error(error.args)
            return {'message': error.args}, HTTP_503_SERVICE_UNAVAILABLE

        return {
            'id': p.pk,
            'reference': p.reference,
            'designation': p.designation,
            'dci': p.dci,
            'desig_bis': p.desig_bis,
            'com_med': p.com_med,
            'risk': p.risque,
            'narcotic': p.stup,
            'stock': stock_info['stock'],
            'stock_by_mag': stock_by_magasin,
            'ucd_cip_list': ucd_cip_list,
            'tiroir_bas': p.tiroir_bas
        }

    def _GetProductStockInfo(self, ref):

        st = (Product
              .select(Product, fn.SUM(Stock.quantite).alias('stock'))
              .join(Stock, on=(Product.reference == Stock.reference))
              .where(Product.reference == ref)
              .group_by(Product))

        if not st:
            stock = 0
        else:
            stock = st[0].stock

        s = (Product
             .select(Product,
                     fn.SUM(Seuil.stock_mini).alias('stock_mini'),
                     fn.SUM(Seuil.stock_maxi).alias('stock_maxi'))
             .join(Seuil, on=(Product.reference == Seuil.reference))
             .group_by(Product))

        return {
            'stock': stock,
            'stock_mini': s and s[0].stock_mini or '-',
            'stock_maxi': s and s[0].stock_maxi or '-'
        }

    def _getStockByMagasin(self, ref):

        mags = (Magasin
                .select(Magasin.mag, Magasin.type_mag, fn.SUM(Stock.quantite).alias('stock'), Magasin.libelle)
                .join(Stock, JOIN.LEFT_OUTER, on=((ref == Stock.reference) & (Magasin.mag == Stock.magasin)))
                # .where(Magasin.eco_type == 'C')
                .group_by(Magasin.mag).order_by(Magasin.eco_type))

        _out = []
        for m in mags:
            if m.type_mag != 'RETOUR':
                q = (FListe.select((fn.SUM(FItem.qte_dem - FItem.qte_serv)).alias('cmd')).join(
                         FItem, on=(
                             (FListe.liste == FItem.liste)
                             & (FListe.mode == FItem.mode)
                             & (FItem.reference == ref)))
                        .where(  # noqa
                             (FListe.mode == 'E')
                             & (FListe.etat == 'V')
                             & (FListe.zone_fin == m.type_mag))
                        .group_by(FListe.zone_fin))

                cmd = q and q[0].cmd or 0
                if cmd < 0:
                    cmd = 0
                _out.append({
                    'mag': m.mag,
                    'type_mag': m.type_mag,
                    'libelle': m.libelle,
                    'stock': m.stock or 0,
                    'commande': cmd or 0,
                })

        return _out

    def _generateProductReference(self):
        while True:
            rand_ref = 'P' + str(round(random.random()*100000000))
            if self._isProductReferenceAvailable(rand_ref):
                break
        return rand_ref

    def _isProductReferenceAvailable(self, ref):
        if (Product.select(Product.reference)
                   .where(Product.reference == ref)
                   .count()) == 0:
            return True
        return False

    def _getProductUCDAndCIPListJSON(self, ref):
        _out = []

        ucd_list = (
            Ucd.select(Ucd.ucd)
            .distinct()
            .where(Ucd.reference == ref)
            .order_by(Ucd.ucd)
        )

        for u in ucd_list:
            ucd_cip_list = Cip.select(Cip.ucd, Cip.cip).distinct().where(Cip.ucd == u.ucd)
            logger.info('Lines : %s.' % len(ucd_cip_list))
            _o = {
                'text': '<b>UCD</b>: ' + u.ucd,
                'children': []
            }
            for c in ucd_cip_list:
                _c = {
                    'text': '<b>CIP</b>: ' + c.cip,
                    'children': []
                }
                cip_version = Cip.select(Cip).where(Cip.ucd == u.ucd, Cip.cip == c.cip)
                for v in cip_version:
                    _v = {
                        'text': 'Version: %s Qt: <b>%s</b> Blist: <b>%s</b> Pass: <b>%s</b>' % (
                            v.dossier, v.qt_boite, v.qt_blister, v.qt_pass
                        ),
                        'icon': 'file',
                        'children': []
                    }
                    _c['children'].append(_v)
                _o['children'].append(_c)
            _out.append(_o)

        return _out
