import os
import polib
import peewee
from common.models import WebLang, WebForm, WebMenu, WebMenuTranslation
from common.util import logger

locales_dir = os.path.join(os.path.dirname(__file__), '..', 'locales')

availables_locales = ['en', 'fr', 'it', 'pl', 'es']


def export_menu():

    return True


def menu_i18n_file_import(fname, langcode=None):
    """Read a PO file and import it on the database"""
    try:
        i18n_file = polib.pofile(fname)
    except OSError:
        logger.warn('File %s for lang %s not found' % (fname, langcode))
        return False

    try:
        lng = WebLang.get(WebLang.iso_639_code == langcode)
    except peewee.DoesNotExist:
        # Lang does not exists in the database, we continue
        return False

    for entry in i18n_file:
        try:
            mnu = WebMenu.get(WebMenu.code == entry.comment)
        except peewee.DoesNotExist:
            logger.warn('Lang %s not found in the database' % (langcode,))
            continue

        tr_name = entry.msgstr
        if not entry.msgstr:
            tr_name = entry.msgid
        try:
            mnu_i18n = WebMenuTranslation()
            mnu_i18n.menu_id = mnu
            mnu_i18n.lang_id = lng
            mnu_i18n.tr_name = tr_name
            mnu_i18n.save()
        except peewee.IntegrityError:
            mnu_i18n = WebMenu.get(WebMenu.code == entry.comment)
            mnu_i18n.tr_name = tr_name
            mnu_i18n.save()

    return True


def form_i18n_file_import(fname, langcode=None):
    """Read a PO file and import it on the database"""
    i18n_file = polib.pofile(fname)
    lng = WebLang.get(WebLang.iso_639_code == langcode)

    for entry in i18n_file:
        tr_name = entry.msgstr
        if not entry.msgstr:
            tr_name = entry.msgid
        try:
            frm_i18n = WebForm()
            frm_i18n.key = entry.comment
            frm_i18n.lang_id = lng
            frm_i18n.tr_name = tr_name
            frm_i18n.save()
        except peewee.IntegrityError:
            frm_i18n = WebForm.get(WebForm.key == entry.comment, WebForm.lang_id == lng)
            frm_i18n.tr_name = tr_name
            frm_i18n.save()

    return True


def form_18n_to_json(langcode):
    """Export translation from the database"""
    res = {}
    try:
        lng = WebLang.get(WebLang.iso_639_code == langcode)
        for term in WebForm.select().where(WebForm.lang_id == lng):
            res[term.key] = term.tr_name
    except peewee.DoesNotExist:
        pass
    return res


def convert_pot2po(pofile, potfile):
    """Merge catalog in the pofile"""
    try:
        po = polib.pofile(pofile)
        pot = polib.pofile(potfile)
        po.merge(pot)
        po.save(pofile)
        return True
    except OSError:
        logger.warn('File %s not found' % (pofile,))
        return False
