from median.base import mysql_db
from median.migration import DbMigrator
from median.models import Profil
from common.models import WebMenu
from peewee import DoesNotExist
from playhouse.dataset import DataSet
import logging
import os
import json

logger = logging.getLogger('median.webserver')

DATA_DIR = os.path.join(os.path.dirname(os.path.realpath(__file__)), 'data')


def check_model(table_name, model_name, importfile=None):
    """Check if table exists, if not we creat it from the model"""
    if not mysql_db.table_exists(table_name):
        logger.warn('Table %s does not exists, we create it from %s' % (table_name, model_name))  # noqa
        print('Table %s does not exists, we create it from %s' % (table_name, model_name))  # noqa
        mysql_db.create_tables([model_name])
        mysql_db.close()
        if importfile:
            # kwargs = {'table_names': [table_name]}
            dsdb = DataSet(mysql_db, table_names=[table_name])
            tbl = dsdb[table_name]
            tbl.thaw(format='json', filename=os.path.join(DATA_DIR, importfile))
            print('Import file %s in table %s' % (importfile, table_name))
    else:
        logger.info('Table %s exists' % (table_name,))
        print('Table %s exists' % (table_name,))

    return True


def sync_menu(table_name, model_name, importfile=None):
    """Check if table exists, if not we creat it from the model"""
    menu_ids = []
    if mysql_db.table_exists(table_name):
        if importfile:
            # we loop around all value, and replace all fields
            js = json.load(open(os.path.join(DATA_DIR, importfile)))
            for j in js:
                menu_ids.append(j['id'])
                try:
                    mnu = WebMenu.get(WebMenu.pk == j['id'])
                    mnu.name = j['name']
                    mnu.code = j['code']
                    mnu.icon = j['icon']
                    mnu.parent = j['parent_id']
                    mnu.sequence = j['seq_order']
                    mnu.url = j['url']
                    mnu.anchor = j['anchor']
                    mnu.save()
                    print('Update menu %s (%s)' % (j['name'], j['id']))
                except DoesNotExist:
                    mnu = WebMenu(
                        pk=j['id'],
                        name=j['name'],
                        code=j['code'],
                        icon=j['icon'],
                        parent=j['parent_id'],
                        sequence=j['seq_order'],
                        url=j['url'],
                        anchor=j['anchor']
                    )
                    mnu.save(force_insert=True)
                    print('Create menu %s (%s)' % (mnu.name, mnu.pk))

            if menu_ids:
                print("Delete old menus")
                q = WebMenu.delete().where(WebMenu.pk.not_in(menu_ids))
                q.execute()
    else:
        logger.info('Table does not exists %s' % (table_name,))
        print('Table does not exists %s' % (table_name,))


def sync_profil():
    """Scan all menus and add on profil for ADMIN and ECO-DEX, DEENOVA"""
    print('Sync profil')
    for prf in ['ECO-DEX', 'DEENOVA', 'ADMIN', 'TECH']:
        print('-> Profil: %s' % prf)
        menus = WebMenu.select()
        for mnu in menus:
            try:
                p = Profil.get(
                    Profil.profil == prf, Profil.ressource == mnu.code
                )
                p.visu = 1
                p.edit = 1
                p.save()
            except DoesNotExist:
                p = Profil(
                    profil=prf,
                    ressource=mnu.code,
                    visu=1,
                    edit=1
                )
                p.save()

    print('Sync profil done')


def median_upgrade():
    """Call the python median Method to add fields on existing model"""
    mig = DbMigrator(mysql_db)
    mig.migrate()
    return True


def median_information():
    """Display Database informations"""
    dbinfo = {
        'name': os.environ.get('MEDIAN_DB_NAME', 'unknown'),
        'server': os.environ.get('MEDIAN_DB_HOST', 'unknown'),
        'user': os.environ.get('MEDIAN_DB_USER', 'unknown'),
        'port': os.environ.get('MEDIAN_DB_PORT', 'unknown'),
    }
    return dbinfo
